/**
 * Copyright (C) 2007-2017 Tatsuo Satoh &lt;multisqllib@gmail.com&gt;
 *
 * This file is part of sqlapp-command.
 *
 * sqlapp-command is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * sqlapp-command is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with sqlapp-command.  If not, see &lt;http://www.gnu.org/licenses/&gt;.
 */

package com.sqlapp.data.db.command.generator;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.poi.ss.usermodel.Workbook;

import com.sqlapp.data.db.command.AbstractTableCommand;
import com.sqlapp.data.db.command.generator.factory.TableGeneratorSettingFactory;
import com.sqlapp.data.db.command.generator.setting.TableGeneratorSetting;
import com.sqlapp.data.db.command.properties.DirectoryProperty;
import com.sqlapp.data.db.command.properties.SqlTypeProperty;
import com.sqlapp.data.db.dialect.Dialect;
import com.sqlapp.data.db.sql.SqlType;
import com.sqlapp.data.schemas.Table;
import com.sqlapp.util.JsonConverter;

import lombok.Getter;
import lombok.Setter;

/**
 * Generate Data and Insert Command
 */
@Getter
@Setter
public class GenerateGeneratorSettingCommand extends AbstractTableCommand
		implements SqlTypeProperty, DirectoryProperty {
	/**
	 * SQL Type
	 */
	private SqlType sqlType = SqlType.INSERT;

	/** file directory */
	private File directory = new File("./");
	/** fileType */
	private GeneratorSettingFileType fileType = GeneratorSettingFileType.EXCEL2007;

	/** TableDataGeneratorSettingFactory */
	private TableGeneratorSettingFactory generatorSettingFactory = new TableGeneratorSettingFactory();

	@Override
	protected void doRun() {
		execute(getDataSource(), connection -> {
			final Dialect dialect = this.getDialect(connection);
			final List<Table> tables = getTables(connection, dialect);
			if (tables.isEmpty()) {
				throw new TableNotFoundException("includeSchemas=" + Arrays.toString(this.getIncludeSchemas())
						+ ", excludeSchemas=" + Arrays.toString(this.getExcludeSchemas()) + ", includeTables="
						+ Arrays.toString(this.getIncludeTables()) + ", excludeTables="
						+ Arrays.toString(this.getExcludeTables()));
			}
			File dir = this.getDirectory();
			if (!dir.exists()) {
				dir.mkdirs();
			}
			for (Table table : tables) {
				writeFile(table, dir, dialect);
			}
		});
	}

	private void writeFile(Table table, File dir, Dialect dialect) throws FileNotFoundException, IOException {
		final TableGeneratorSetting setting = this.getGeneratorSettingFactory().createDefault(table, dialect,
				this.getTableOptions(), this.getSqlType());
		switch (this.getFileType()) {
		case JSON:
		case YAML:
			writeTextFile(setting, dir);
			break;
		default:
			writeFileWorkbook(setting, dir);
		}
	}

	private void writeTextFile(TableGeneratorSetting setting, File dir) throws FileNotFoundException, IOException {
		JsonConverter jsonConverter = this.getFileType().getWorkbookFileType().createJsonConverter();
		jsonConverter.setIndentOutput(true);
		String text = jsonConverter.toJsonString(setting);
		FileUtils.write(
				new File(dir, setting.getName() + "." + this.getFileType().getWorkbookFileType().getFileExtension()),
				text, Charset.forName("UTF-8"));
	}

	private void writeFileWorkbook(TableGeneratorSetting setting, File dir) throws FileNotFoundException, IOException {
		try (Workbook wb = this.getFileType().getWorkbookFileType().createWorkbook()) {
			GeneratorSettingWorkbook.Table.writeSheet(setting, wb);
			GeneratorSettingWorkbook.Column.writeSheet(setting, wb);
			GeneratorSettingWorkbook.Query.writeSheet(setting, wb);
			File file = new File(dir, setting.getName() + ".xlsx");
			try (FileOutputStream os = new FileOutputStream(file);
					BufferedOutputStream bs = new BufferedOutputStream(os)) {
				wb.write(bs);
				bs.flush();
			}
		}
	}

}
