/**
 * Copyright (C) 2007-2017 Tatsuo Satoh <multisqllib@gmail.com>
 *
 * This file is part of sqlapp-command.
 *
 * sqlapp-command is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * sqlapp-command is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with sqlapp-command.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sqlapp.data.db.command.html;

import java.io.File;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.sqlapp.data.schemas.Catalog;
import com.sqlapp.data.schemas.Column;
import com.sqlapp.data.schemas.ForeignKeyConstraint;
import com.sqlapp.data.schemas.Schema;
import com.sqlapp.data.schemas.Table;
import com.sqlapp.exceptions.InvalidTextException;
import com.sqlapp.util.CommonUtils;
import com.sqlapp.util.FileUtils;

import lombok.Data;

public class VirtualForeignKeyLoader {
	
	private String encoding="utf8";
	
	public VirtualForeignKeyLoader(){
	}

	public void load(Catalog catalog, File file){
		if (file==null||!file.exists()){
			return;
		}
		for(File child:file.listFiles()){
			List<String> texts=FileUtils.readTextList(child, encoding);
			loadInternal(catalog, texts);
		}
	}

	private void loadInternal(Catalog catalog, List<String> texts){
		for(int i=0;i<texts.size();i++){
			String text=texts.get(i);
			if (isComment(text)){
				continue;
			}
			TablePair pair=parse(text, i+1);
			Table from=getTable(pair, pair.getFrom(), catalog);
			Table to=getTable(pair, pair.getTo(), catalog);
			Column[] columns=getColumns(pair, pair.getFrom(), from);
			Column[] pkColumns=getColumns(pair, pair.getTo(), to);
			ForeignKeyConstraint fk=new ForeignKeyConstraint("fk_"+from.getName()+"_virtual"+(from.getConstraints().getForeignKeyConstraints().size()+1), columns, pkColumns);
			fk.setVirtual(true);
			from.getConstraints().add(fk);
		}
	}

	private Table getTable(TablePair pair, Table table,Catalog catalog){
		Table from;
		if (!CommonUtils.isEmpty(table.getSchemaName())){
			Schema schema=catalog.getSchemas().get(pair.getFrom().getSchemaName());
			if (schema==null){
				throw new InvalidTextException(pair.getLine(), pair.getLineNo(), table+"(Schema) does not found.");
			}
			from=schema.getTable(table.getName());
			if (from==null){
				throw new InvalidTextException(pair.getLine(), pair.getLineNo(), table+" does not found.");
			}
			return from;
		} else{
			for(Schema schema:catalog.getSchemas()){
				from=schema.getTable(table.getName());
				if (from!=null){
					return from;
				}
			}
		}
		throw new InvalidTextException(pair.getLine(), pair.getLineNo(), table+" does not found.");
	}
	
	private Column[] getColumns(TablePair pair, Table ref, Table table){
		List<Column> columns=CommonUtils.list();
		if (ref.getColumns().isEmpty()){
			for(Column column:table.getColumns()){
				if (column.isPrimaryKey()){
					columns.add(column);
				}
			}
		} else{
			for(Column col:ref.getColumns()){
				Column column=table.getColumns().get(col.getName());
				if (column==null){
					throw new InvalidTextException(pair.getLine(), pair.getLineNo(), col+" does not found.");
				}
				columns.add(column);
			}
		}
		return columns.toArray(new Column[0]);
	}
	

	private static Pattern COMMENT_PATTERN=Pattern.compile("\\s*#.*");

	private boolean isComment(String text){
		Matcher matcher=COMMENT_PATTERN.matcher(text);
		return matcher.matches();
	}
	
	private TablePair parse(String text, int lineNo){
		String base=text;
		text=text.trim();
		String[] texts=text.split("\\s*->\\s*");
		if (texts.length==0){
			throw new InvalidTextException(base, lineNo, "No relations(->) found.");
		} if (texts.length!=2){
			throw new InvalidTextException(base, lineNo, "Multiple relations(->) found. count="+texts.length);
		}
		Table from=parseTable(base, texts[0], lineNo);
		Table to=parseTable(base, texts[1], lineNo);
		TablePair pair=new TablePair();
		pair.setFrom(from);
		pair.setTo(to);
		pair.setLine(text);
		pair.setLineNo(lineNo);
		return pair;
	}

	private Table parseTable(String base, String tableText, int lineNo){
		int start=tableText.indexOf('(');
		String tablePart=null;
		Table table=new Table();
		if (start>0){
			tablePart=tableText.substring(0, start).trim();
			if (!tableText.endsWith(")")){
				throw new InvalidTextException(base, lineNo, "aaaaa(id,val)->bbbbb");
			}
			String[] columns=tableText.substring(start+1, tableText.length()-1).split("\\s*,\\s*");
			for(String col:columns){
				Column column=table.newColumn();
				col=col.trim();
				if (CommonUtils.isEmpty(col)){
					throw new InvalidTextException(base, lineNo, "Invalid column definition. value="+tableText);
				}
				column.setName(col);
				table.getColumns().add(column);
			}
		} else{
			tablePart=tableText;
		}
		String[] names=tablePart.split("\\.");
		if (names.length==1){
			table.setName(names[0]);
		} else if (names.length==2){
			table.setSchemaName(names[0]);
			table.setName(names[1]);
		} else if (names.length==2){
			table.setCatalogName(names[0]);
			table.setSchemaName(names[1]);
			table.setName(names[2]);
		} else{
			throw new InvalidTextException(base, lineNo, "Invalid tableName. value="+tablePart);
		}
		return table;
	}

	@Data
	static class TablePair{
		private Table from;
		private Table to;
		private String line;
		private int lineNo;
	}
	
}
