/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation; either version
 * 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 */

package com.wedeploy.api.sdk;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
public abstract class ErrorData<T> {

	/**
	 * Creates body string.
	 */
	public String errorBody() {
		StringBuilder errorBody = new StringBuilder();

		errorBody.append("{\n\t\"code\": ");
		errorBody.append(statusCode);
		errorBody.append(",\n");
		errorBody.append("\t\"message\": \"");
		errorBody.append(encodeString(statusMessage));
		errorBody.append("\"");

		if (!subErrors.isEmpty()) {
			errorBody.append(",\n");
			errorBody.append("\t\"errors\": [\n");

			for (int i = 0; i < subErrors.size(); i++) {
				String[] subError = subErrors.get(i);

				if (i != 0) {
					errorBody.append(",\n");
				}

				errorBody.append("\t\t{\n");
				errorBody.append("\t\t\t\"reason\": \"");
				errorBody.append(encodeString(subError[0]));
				if (subError.length >= 2) {
					errorBody.append("\",\n");
					errorBody.append("\t\t\t\"message\": \"");
					errorBody.append(encodeString(subError[1]));
					errorBody.append("\"");

					if (subError.length >= 3 && subError[2] != null) {
						errorBody.append(",\n");
						errorBody.append("\t\t\t\"details\": \"");
						errorBody.append(encodeString(subError[2]));
						errorBody.append("\"\n");
					}
				}
				errorBody.append("\t\t}");
			}

			errorBody.append("\n\t]\n");
		}
		else {
			errorBody.append("\n");
		}

		errorBody.append("}");

		return errorBody.toString();
	}

	public void headers(BiConsumer<String, String> consumer) {
		headers.forEach(consumer);
	}

	/**
	 * Returns status code.
	 */
	public int statusCode() {
		return statusCode;
	}

	/**
	 * Returns status message.
	 */
	public String statusMessage() {
		return statusMessage;
	}

	/**
	 * Adds error reason and a message.
	 * @see #add(String, String, Throwable)
	 */
	protected void add(String reason, String message) {
		subErrors.add(new String[] {reason, message});
	}
	/**
	 * Adds error reason, message and throwable details.
	 * @see #add(String, String)
	 */
	protected void add(String reason, String message, Throwable throwable) {
		subErrors.add(new String[] {reason, message, throwableToString(throwable)});
	}

	protected List<String[]> getSubErrors() {
		return subErrors;
	}

	protected void header(String name, String value) {
		headers.put(name, value);
	}

	/**
	 * Writes the response into the target.
	 */
	protected abstract T into(T targetConsumer);

	protected void set(
		int statusCode, String statusMessage, String defaultMessage) {

		this.statusCode = statusCode;

		if (statusMessage == null) {
			statusMessage = defaultMessage;
		}

		this.statusMessage = statusMessage;
	}

	/**
	 * Encodes input to JSON-safe string.
	 */
	private String encodeString(String value) {
		StringBuilder sb = new StringBuilder();

		int len = value.length();

		for (int i = 0; i < len; i++) {
			char c = value.charAt(i);

			switch (c) {
				case '"':
					sb.append("\\\"");
					break;
				case '\\':
					sb.append("\\\\");
					break;
				case '/':
					sb.append("\\/");
					break;
				case '\b':
					sb.append("\\b");
					break;
				case '\f':
					sb.append("\\f");
					break;
				case '\n':
					sb.append("\\n");
					break;
				case '\r':
					sb.append("\\r");
					break;
				case '\t':
					sb.append("\\t");
					break;
				default:
					sb.append(c);
			}
		}

		return sb.toString();
	}

	protected static String throwableToString(Throwable throwable) {
		if (throwable == null) {
			return null;
		}
		StringBuilder sb = new StringBuilder();

		sb.append(throwable.toString());
		sb.append("\n");

		while (throwable != null) {
			sb.append("---\n");
			StackTraceElement[] stackTraceElements = throwable.getStackTrace();

			int len = stackTraceElements.length >= 3 ? 3 : stackTraceElements.length;

			for (int i = 0; i < len; i++) {
				sb.append(stackTraceElements[i].toString());
				sb.append("\n");
			}

			if (stackTraceElements.length > 3 ) {
				sb.append("...\n");
			}

			throwable = throwable.getCause();
		}

		return sb.toString();
	}

	private final Map<String, String> headers = new HashMap<>();
	private int statusCode;
	private String statusMessage;
	private final LinkedList<String[]> subErrors = new LinkedList<>();

}
