/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * <p>
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * <p>
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * <p>
 * 3. Neither the name of Liferay, Inc. nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package com.wedeploy.api.sdk;

import java.util.List;
import java.util.Map;

/**
 * HTTP request.
 */
public interface Request {

	/**
	 * Returns the auth.
	 */
	public Auth auth();

	/**
	 * Returns the base URL corresponding to the the HTTP request.
	 * If <code>null</code> the request represents a 'local' call,
	 * with only a {@link #path()} existing.
	 */
	public String baseUrl();

	/**
	 * Returns the body content as a string.
	 */
	public String body();

	/**
	 * Sets the raw body content.
	 */
	public Request body(byte[] body);

	/**
	 * Sets the body content, assuming JSON content type.
	 */
	public Request body(Object body);

	/**
	 * Sets the body content by passing the string value.
	 */
	public Request body(String body);

	/**
	 * Sets the raw body content and the {@link ContentType content-type}.
	 * This is just a shortcut method.
	 */
	public Request body(String body, ContentType contentType);

	/**
	 * Returns the raw body content.
	 */
	public byte[] bodyBytes();

	/**
	 * Returns parsed {@link #body() body content} as a List.
	 */
	public <T> List<T> bodyList(Class<T> componentType);

	/**
	 * Returns parsed {@link #body() body content}.
	 * If body is not set, returns {@code null}.
	 */
	public <K, V> Map<K, V> bodyMap(Class<K> keyType, Class<V> valueType);

	/**
	 * REturns parsed {@link #body() body content}.
	 * If body is not set, returns {@code null}.
	 */
	public default <V> Map<String, V> bodyMap(Class<V> valueType) {
		return bodyMap(String.class, valueType);
	}

	/**
	 * Parses the body depending on content-type. If content-type is NOT set,
	 * it will use assume the "plain/text" content type.
	 * Returns parsed {@link #body() body content}.
	 * If body is not set, returns <code>null</code>.
	 * If body can not be parsed, throws an Exception.
	 */
	public <T> T bodyValue();

	/**
	 * Parses the body depending on content-type into the target type.
	 *
	 * @see #bodyValue()
	 */
	public <T> T bodyValue(Class<T> type);

	/**
	 * Gets the content type header.
	 */
	public String contentType();

	/**
	 * Sets the content type header.
	 */
	public Request contentType(ContentType contentType);

	/**
	 * Returns the context that this request belongs to.
	 */
	public Context context();

	/**
	 * @return A map of all the cookies.
	 */
	Map<String, Cookie> cookies();

	/**
	 * Adds a cookie to the response.
	 */
	public Request cookie(Cookie cookie);

	/**
	 * Gets cookie value by name.
	 */
	public Cookie cookie(String name);

	/**
	 * Returns array of uploaded files. Returns <code>null</code> if nothing
	 * was uploaded.
	 */
	public FileUpload[] fileUploads();

	/**
	 * Returns the form parameter.
	 */
	public Object form(String name);

	/**
	 * Sets the form parameter.
	 */
	public Request form(String name, Object value);

	/**
	 * Returns form parameters.
	 */
	public MultiMap<Object> forms();

	/**
	 * Gets header value by name.
	 */
	public String header(String name);

	/**
	 * Sets the first header name with the specified value.
	 */
	public Request header(String name, String value);

	/**
	 * Returns headers.
	 */
	public MultiMap<String> headers();

	/**
	 * Checks if the request contains a content type header with the
	 * specified value.
	 */
	public boolean isContentType(ContentType contentType);

	/**
	 * Returns HTTP method, in uppercase.
	 */
	public String method();

	/**
	 * Sets method.
	 */
	public Request method(String method);

	/**
	 * Delegates this request to the next handler.
	 */
	public void next();

	/**
	 * Delegates this request to the next handler and executes
	 * the runnable on the response end.
	 */
	public void next(OnResponseEndCallback onResponseEnd);

	/**
	 * Gets parameter value by name.
	 */
	public String param(String name);

	/**
	 * Sets parameter value.
	 */
	public Request param(String name, String value);

	/**
	 * Returns map of string request parameters. They consist of:
	 * <ol>
	 * <li>query parameters</li>
	 * <li>form parameters</li>
	 * <li>parameters from api.json (serialized to string)</li>
	 * <li>path parameters</li>
	 * </ol>
	 * <p>
	 * Parameters are accumulated in given order.
	 */
	public MultiMap<String> params();

	/**
	 * Returns action path of the URI. Returned path does not contain the query
	 * part. Returns <code>"/"</code> for the root.
	 */
	public String path();

	/**
	 * Returns the query string of the URL.
	 */
	public String query();

	/**
	 * Returns associated response of this request.
	 */
	public Response response();

	/**
	 * Associates response to this request.
	 */
	public void response(Response response);

	/**
	 * Returns current session.
	 */
	public Session session();

	/**
	 * Returns the absolute URL corresponding to the the HTTP request.
	 */
	public String url();

	/**
	 * Merges all the inputs in best possible way. The following is merged
	 * in given order:
	 * <ol>
	 * <li>{@link #params() parameters} - parsed to a JSON, if possible</li>
	 * <li>{@link #bodyValue()} parsed body}</li>
	 * </ol>
	 * Values with the same names and different sources are accumulated in
	 * multi map. Always returns an object, never a <code>null</code>.
	 * When content type is one of the form content types, body is ignored.
	 */
	public MultiMap<Object> values();

	/**
	 * Returns merged values to a certain type.
	 *
	 * @see #values()
	 */
	public <T> T values(Class<T> type);

}
