package ch.framedev.javasqliteutils;

import java.io.File;
import java.sql.*;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This Plugin was Created by FrameDev
 * Package : mysql
 * Date: 07.03.21
 * Project: untitled
 * Copyrighted by FrameDev
 */
@SuppressWarnings({"unused", "ResultOfMethodCallIgnored"})
public class SQLiteV2 {
    private final Logger LOGGER = Logger.getLogger(SQLiteV2.class.getName());

    public Connection connection;
    private final String fileName;
    private final String path;

    private final ExecutorService executor = Executors.newFixedThreadPool(10);

    public static class Builder {
        private String fileName;
        private String path;

        public Builder setFileName(String fileName) {
            this.fileName = fileName;
            return this;
        }

        public Builder setPath(String path) {
            this.path = path;
            return this;
        }

        public SQLiteV2 build() {
            if (fileName == null || path == null) {
                throw new IllegalArgumentException("File name and path must be set.");
            }
            new File(path).mkdirs(); // Ensure the directory exists
            return new SQLiteV2(path, fileName);
        }
    }

    protected SQLiteV2(String path, String database) {
        this.path = path;
        this.fileName = database;
    }

    public void connectAsync(Callback<Connection> callback) {
        executor.execute(() -> {
            try {
                if (connection != null && !connection.isClosed()) {
                    callback.onResult(connection);
                    return;
                }
                Class.forName("org.sqlite.JDBC");
                String url = "jdbc:sqlite:" + path + "/" + fileName;
                connection = DriverManager.getConnection(url);
                callback.onResult(connection);
            } catch (SQLException | ClassNotFoundException e) {
                callback.onError(e);
            }
        });
    }

    public Connection connect() {
        try {
            if (connection != null && !connection.isClosed()) {
                return connection;
            }
            Class.forName("org.sqlite.JDBC");
            String url = "jdbc:sqlite:" + path + "/" + fileName;
            connection = DriverManager.getConnection(url);
            return connection;
        } catch (SQLException | ClassNotFoundException e) {
            throw new RuntimeException("Failed to connect to SQLite database", e);
        }
    }

    public void close() {
        if (connection != null) {
            try {
                if (!connection.isClosed()) {
                    connection.close();
                }
            } catch (SQLException ex) {
                LOGGER.log(Level.SEVERE, "Failed to close SQLite connection", ex);
            } finally {
                connection = null; // Always nullify the reference
            }
        }
    }

    /**
     * Checks if a table exists in the SQLite database.
     *
     * @param table The name of the table to check.
     * @return true if the table exists, false otherwise.
     */
    public boolean isTableExists(String table) {
        String sql = "SELECT name FROM sqlite_master WHERE type = 'table' AND name = ? AND name NOT LIKE 'sqlite_%';";
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            pstmt.setString(1, table);
            try (ResultSet rs = pstmt.executeQuery()) {
                return rs.next(); // Returns true if the table exists, false otherwise
            }

        } catch (SQLException ex) {
            LOGGER.log(Level.SEVERE, "Failed to check if table exists: " + table, ex);
        }
        return false;
    }

    /**
     * Asynchronously checks if a table exists in the SQLite database.
     *
     * @param table    The name of the table to check.
     * @param callback The callback to handle the result.
     */
    public void isTableExistsAsync(String table, Callback<Boolean> callback) {
        String sql = "SELECT name FROM sqlite_master WHERE type = 'table' AND name = ? AND name NOT LIKE 'sqlite_%';";

        executor.execute(() -> connectAsync(new Callback<Connection>() {
            @Override
            public void onResult(Connection result) {
                try (Connection conn = result;
                     PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                    pstmt.setString(1, table);
                    try (ResultSet rs = pstmt.executeQuery()) {
                        callback.onResult(rs.next()); // Returns true if the table exists, false otherwise
                    }

                } catch (SQLException ex) {
                    callback.onError(ex);
                }
            }

            @Override
            public void onError(Exception e) {
                callback.onError(e);
            }
        }));
    }
    
    /**
     * Creates a table with the specified name and columns.
     *
     * @param tableName The name of the table to create.
     * @param date      Whether to include a date column.
     * @param columns   The columns to include in the table.
     * @return true if the table was created successfully, false otherwise.
     */
    public boolean createTable(String tableName, boolean date, String[] columns) {
        String sql = getString(tableName, date, columns);
        try (Connection conn = connect();
             Statement stmt = conn.createStatement()) {
            return stmt.execute(sql);
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to create table: " + tableName, e);
        }
        return false;
    }
    
    /**
     * Asynchronously creates a table with the specified name and columns.
     *
     * @param tableName The name of the table to create.
     * @param date      Whether to include a date column.
     * @param columns   The columns to include in the table.
     * @param callback  The callback to handle the result.
     */
    public void createTableAsync(String tableName, boolean date, String[] columns, Callback<Boolean> callback) {
        String sql = getString(tableName, date, columns);

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         Statement stmt = conn.createStatement()) {
                        stmt.execute(sql);
                        callback.onResult(true);
                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Constructs the SQL string for creating a table.
     *
     * @param tableName The name of the table.
     * @param date      Whether to include a date column.
     * @param columns   The columns to include in the table.
     * @return The SQL string for creating the table.
     */
    private static String getString(String tableName, boolean date, String[] columns) {
        StringBuilder stringBuilder = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            stringBuilder.append(columns[i]);
            if (i < columns.length - 1) {
                stringBuilder.append(",");
            }
        }

        String sql;
        if (date) {
            sql = "CREATE TABLE IF NOT EXISTS " + tableName +
                  " (ID INTEGER PRIMARY KEY AUTOINCREMENT, " +
                  stringBuilder.toString() +
                  ", created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP);";
        } else {
            sql = "CREATE TABLE IF NOT EXISTS " + tableName +
                  " (ID INTEGER PRIMARY KEY AUTOINCREMENT, " +
                  stringBuilder.toString() +
                  ");";
        }
        return sql;
    }

    /**
     * Inserts data into a specified table with the given columns.
     *
     * @param table  The name of the table to insert data into.
     * @param data   The data to insert.
     * @param columns The columns corresponding to the data.
     * @return true if the insertion was successful, false otherwise.
     */
    public boolean insertData(String table, Object[] data, String[] columns) {
        if (columns.length == 0 || data.length == 0 || columns.length != data.length) {
            throw new IllegalArgumentException("Columns and data arrays must not be empty and must have the same length.");
        }

        // Join the column names with commas
        String columnNames = String.join(",", columns);

        // Create a string of placeholders (e.g., "?, ?, ?")
        String placeholders = String.join(",", Collections.nCopies(columns.length, "?"));

        // Construct the SQL insert statement
        String sql = "INSERT INTO " + table + " (" + columnNames + ") VALUES (" + placeholders + ")";
        try (Connection conn = connect();
             PreparedStatement stmt = conn.prepareStatement(sql)) {

            // Set the values for each placeholder
            for (int i = 0; i < data.length; i++) {
                stmt.setObject(i + 1, data[i]);
            }

            // Execute the statement
            return stmt.executeUpdate() > 0;

        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to insert data into table: " + table, e);
        }
        return false;
    }

    /**
     * Asynchronously inserts data into a specified table with the given columns.
     *
     * @param table   The name of the table to insert data into.
     * @param data    The data to insert.
     * @param columns The columns corresponding to the data.
     * @param callback The callback to handle the result.
     */
    public void insertDataAsync(String table, Object[] data, String[] columns, Callback<Boolean> callback) {
        if (columns.length == 0 || data.length == 0 || columns.length != data.length) {
            throw new IllegalArgumentException("Columns and data arrays must not be empty and must have the same length.");
        }

        // Join the column names with commas
        String columnNames = String.join(",", columns);

        // Create a string of placeholders (e.g., "?, ?, ?")
        String placeholders = String.join(",", Collections.nCopies(columns.length, "?"));

        // Construct the SQL insert statement
        String sql = "INSERT INTO " + table + " (" + columnNames + ") VALUES (" + placeholders + ")";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement stmt = conn.prepareStatement(sql)) {

                        // Set the values for each placeholder
                        for (int i = 0; i < data.length; i++) {
                            stmt.setObject(i + 1, data[i]);
                        }

                        // Execute the statement
                        stmt.executeUpdate();
                        callback.onResult(true);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Updates data in a specified table.
     *
     * @param table    The name of the table to update.
     * @param selected The column to update.
     * @param data     The new value for the column.
     * @param where    The condition for the update (e.g., "id = 1").
     * @return true if the update was successful, false otherwise.
     */
    public boolean updateData(String table, String selected, Object data, String where) {
        String sql = "UPDATE " + table + " SET " + selected + " = ? WHERE " + where;
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            // Set the value for the placeholder
            pstmt.setObject(1, data);

            // Execute the update
            return pstmt.executeUpdate() > 0;

        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to update data in table: " + table, e);
        }
        return false;
    }

    /**
     * Asynchronously updates data in a specified table.
     *
     * @param table    The name of the table to update.
     * @param selected The column to update.
     * @param data     The new value for the column.
     * @param where    The condition for the update (e.g., "id = 1").
     * @param callback The callback to handle the result.
     */
    public void updateDataAsync(String table, String selected, Object data, String where, Callback<Boolean> callback) {
        String sql = "UPDATE " + table + " SET " + selected + " = ? WHERE " + where;

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        // Set the value for the placeholder
                        pstmt.setObject(1, data);

                        // Execute the update
                        pstmt.executeUpdate();
                        callback.onResult(pstmt.executeUpdate() > 0);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Updates data in a specified table with multiple columns.
     *
     * @param table    The name of the table to update.
     * @param columns  The columns to update.
     * @param values   The new values for the columns.
     * @param where    The condition for the update (e.g., "id = 1").
     * @return true if the update was successful, false otherwise.
     */
    public boolean updateData(String table, String[] columns, Object[] values, String where) {
        if (columns.length == 0 || values.length == 0 || columns.length != values.length) {
            throw new IllegalArgumentException("Columns and values arrays must not be empty and must have the same length.");
        }

        // Construct the SET clause dynamically
        StringBuilder setClause = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            setClause.append(columns[i]).append(" = ?");
            if (i < columns.length - 1) {
                setClause.append(", ");
            }
        }

        String sql = "UPDATE " + table + " SET " + setClause.toString() + " WHERE " + where;
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            // Set the values for the placeholders
            for (int i = 0; i < values.length; i++) {
                pstmt.setObject(i + 1, values[i]);
            }

            // Execute the update
            return pstmt.executeUpdate() > 0;

        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to update data in table: " + table, e);
        }
        return false;
    }

    /**
     * Asynchronously updates data in a specified table with multiple columns.
     *
     * @param table    The name of the table to update.
     * @param columns  The columns to update.
     * @param values   The new values for the columns.
     * @param where    The condition for the update (e.g., "id = 1").
     * @param callback The callback to handle the result.
     */
    public void updateDataAsync(String table, String[] columns, Object[] values, String where, Callback<Boolean> callback) {
        if (columns.length == 0 || values.length == 0 || columns.length != values.length) {
            throw new IllegalArgumentException("Columns and values arrays must not be empty and must have the same length.");
        }

        // Construct the SET clause dynamically
        StringBuilder setClause = new StringBuilder();
        for (int i = 0; i < columns.length; i++) {
            setClause.append(columns[i]).append(" = ?");
            if (i < columns.length - 1) {
                setClause.append(", ");
            }
        }

        String sql = "UPDATE " + table + " SET " + setClause.toString() + " WHERE " + where;

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        // Set the values for the placeholders
                        for (int i = 0; i < values.length; i++) {
                            pstmt.setObject(i + 1, values[i]);
                        }

                        // Execute the update
                        callback.onResult(pstmt.executeUpdate() > 0);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Deletes data from a specified table based on a condition.
     *
     * @param table The name of the table to delete data from.
     * @param where The condition for the deletion (e.g., "id = 1").
     * @return true if the deletion was successful, false otherwise.
     */
    public boolean deleteDataInTable(String table, String where) {
        String sql = "DELETE FROM " + table + " WHERE " + where;
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            // Execute the delete
            return pstmt.executeUpdate() > 0;

        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to delete data in table: " + table, e);
        }
        return false;
    }

    /**
     * Asynchronously deletes data from a specified table based on a condition.
     *
     * @param table    The name of the table to delete data from.
     * @param where    The condition for the deletion (e.g., "id = 1").
     * @param callback The callback to handle the result.
     */
    public void deleteDataInTableAsync(String table, String where, Callback<Boolean> callback) {
        String sql = "DELETE FROM " + table + " WHERE " + where;

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        // Execute the delete
                        callback.onResult(pstmt.executeUpdate() > 0);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Deletes data from a specified table based on two conditions.
     *
     * @param table       The name of the table to delete data from.
     * @param whereColumn The column for the first condition.
     * @param whereValue  The value for the first condition.
     * @param andColumn   The column for the second condition.
     * @param andValue    The value for the second condition.
     * @return true if the deletion was successful, false otherwise.
     */
    public boolean deleteDataInTable(String table, String whereColumn, String whereValue, String andColumn, String andValue) {
        String sql = "DELETE FROM " + table + " WHERE " + whereColumn + " = ? AND " + andColumn + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            pstmt.setString(1, whereValue);
            pstmt.setString(2, andValue);

            // Execute the delete
            return pstmt.executeUpdate() > 0;

        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to delete data in table: " + table, e);
        }
        return false;
    }

    /**
     * Asynchronously deletes data from a specified table based on two conditions.
     *
     * @param table       The name of the table to delete data from.
     * @param whereColumn The column for the first condition.
     * @param whereValue  The value for the first condition.
     * @param andColumn   The column for the second condition.
     * @param andValue    The value for the second condition.
     * @param callback    The callback to handle the result.
     */
    public void deleteDataInTableAsync(String table, String whereColumn, String whereValue, String andColumn, String andValue, Callback<Boolean> callback) {
        String sql = "DELETE FROM " + table + " WHERE " + whereColumn + " = ? AND " + andColumn + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        pstmt.setString(1, whereValue);
                        pstmt.setString(2, andValue);

                        // Execute the delete
                        callback.onResult(pstmt.executeUpdate() > 0);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Checks if a record exists in a specified table based on a column and data.
     *
     * @param table  The name of the table to check.
     * @param column The column to check.
     * @param data   The data to check for existence.
     * @return true if the record exists, false otherwise.
     */
    public boolean exists(String table, String column, String data) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            pstmt.setString(1, data);
            try (ResultSet res = pstmt.executeQuery()) {
                return res.next(); // Returns true if the record exists
            }

        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to check if record exists in table: " + table, e);
        }
        return false;
    }

    /**
     * Asynchronously checks if a record exists in a specified table based on a column and data.
     *
     * @param table    The name of the table to check.
     * @param column   The column to check.
     * @param data     The data to check for existence.
     * @param callback The callback to handle the result.
     */
    public void existsAsync(String table, String column, String data, Callback<Boolean> callback) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        pstmt.setString(1, data);
                        try (ResultSet res = pstmt.executeQuery()) {
                            callback.onResult(res.next()); // Returns true if the record exists
                        }

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Checks if a record exists in a specified table based on two conditions.
     *
     * @param table       The name of the table to check.
     * @param column      The column to check.
     * @param data        The data to check for existence.
     * @param andColumn   The second column for the condition.
     * @param andValue    The value for the second condition.
     * @return true if the record exists, false otherwise.
     */
    public boolean exists(String table, String column, String data, String andColumn, String andValue) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ? AND " + andColumn + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            pstmt.setString(1, data);
            pstmt.setString(2, andValue);

            try (ResultSet res = pstmt.executeQuery()) {
                return res.next(); // Returns true if the record exists
            }

        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to check if record exists in table: " + table, e);
        }
        return false;
    }

    /**
     * Asynchronously checks if a record exists in a specified table based on two conditions.
     *
     * @param table       The name of the table to check.
     * @param column      The column to check.
     * @param data        The data to check for existence.
     */
    public boolean exists(String table, String[] column, String[] data) {
        if (column == null || data == null || column.length == 0 || data.length == 0 || column.length != data.length) {
            throw new IllegalArgumentException("Columns and data arrays must not be empty and must have the same length.");
        }

        // Build WHERE clause: col1 = ? AND col2 = ? ...
        StringBuilder whereClause = new StringBuilder();
        for (int i = 0; i < column.length; i++) {
            whereClause.append(column[i]).append(" = ?");
            if (i < column.length - 1) {
                whereClause.append(" AND ");
            }
        }

        String sql = "SELECT * FROM " + table + " WHERE " + whereClause;
        try (Connection conn = connect();
             PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

            for (int i = 0; i < data.length; i++) {
                pstmt.setString(i + 1, data[i]);
            }

            try (ResultSet res = pstmt.executeQuery()) {
                return res.next();
            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to check if record exists in table: " + table, e);
        }
        return false;
    }

    /**
     * Asynchronously checks if a record exists in a specified table based on two conditions.
     *
     * @param table       The name of the table to check.
     * @param column      The column to check.
     * @param data        The data to check for existence.
     * @param andColumn   The second column for the condition.
     * @param andValue    The value for the second condition.
     * @param callback    The callback to handle the result.
     */
    public void existsAsync(String table, String column, String data, String andColumn, String andValue, Callback<Boolean> callback) {
        String sql = "SELECT * FROM " + table + " WHERE " + column + " = ? AND " + andColumn + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (Connection conn = result;
                         PreparedStatement pstmt = Objects.requireNonNull(conn).prepareStatement(sql)) {

                        pstmt.setString(1, data);
                        pstmt.setString(2, andValue);

                        try (ResultSet res = pstmt.executeQuery()) {
                            callback.onResult(res.next()); // Returns true if the record exists
                        }

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Retrieves a single value from a specified table based on a column and data.
     *
     * @param table    The name of the table to query.
     * @param selected The column to select.
     * @param column   The column to filter by.
     * @param data     The value to filter by.
     * @return The value from the selected column, or null if not found.
     */
    public Object get(String table, String selected, String column, Object data) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            pstmt.setObject(1, data);

            try (ResultSet res = pstmt.executeQuery()) {
                if (res.next()) {
                    return res.getObject(selected);
                }
            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Database query failed: " + sql, e);
        }
        return null;
    }

    /**
     * Asynchronously retrieves a single value from a specified table based on a column and data.
     *
     * @param table    The name of the table to query.
     * @param selected The column to select.
     * @param column   The column to filter by.
     * @param data     The value to filter by.
     */
    public Object get(String table, String selected, String[] column, Object[] data) {
        if (column == null || data == null || column.length == 0 || data.length == 0 || column.length != data.length) {
            throw new IllegalArgumentException("Columns and data arrays must not be empty and must have the same length.");
        }

        // Build WHERE clause: col1 = ? AND col2 = ? ...
        StringBuilder whereClause = new StringBuilder();
        for (int i = 0; i < column.length; i++) {
            whereClause.append(column[i]).append(" = ?");
            if (i < column.length - 1) {
                whereClause.append(" AND ");
            }
        }

        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + whereClause;
        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            for (int i = 0; i < data.length; i++) {
                pstmt.setObject(i + 1, data[i]);
            }

            try (ResultSet res = pstmt.executeQuery()) {
                if (res.next()) {
                    return res.getObject(selected);
                }
            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Database query failed: " + sql, e);
        }
        return null;
    }

    public Optional<Object> getOptional(String table, String selected, String[] column, Object[] data) {
        if (column == null || data == null || column.length == 0 || data.length == 0 || column.length != data.length) {
            throw new IllegalArgumentException("Columns and data arrays must not be empty and must have the same length.");
        }

        // Build WHERE clause: col1 = ? AND col2 = ? ...
        StringBuilder whereClause = new StringBuilder();
        for (int i = 0; i < column.length; i++) {
            whereClause.append(column[i]).append(" = ?");
            if (i < column.length - 1) {
                whereClause.append(" AND ");
            }
        }

        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + whereClause;
        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            for (int i = 0; i < data.length; i++) {
                pstmt.setObject(i + 1, data[i]);
            }

            try (ResultSet res = pstmt.executeQuery()) {
                if (res.next()) {
                    return Optional.ofNullable(res.getObject(selected));
                }
            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Database query failed: " + sql, e);
        }
        return Optional.empty();
    }


    /**
     * Asynchronously retrieves a single value from a specified table based on a column and data.
     *
     * @param table    The name of the table to query.
     * @param selected The column to select.
     * @param column   The column to filter by.
     * @param data     The value to filter by.
     * @param callback The callback to handle the result.
     */
    public void getAsync(String table, String selected, String column, Object data, Callback<Object> callback) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (PreparedStatement pstmt = result.prepareStatement(sql)) {

                        pstmt.setObject(1, data);

                        try (ResultSet res = pstmt.executeQuery()) {
                            if (res.next()) {
                                callback.onResult(res.getObject(selected));
                            }
                        }
                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Retrieves multiple values from a specified table based on a column and data.
     *
     * @param table           The name of the table to query.
     * @param selectedColumns The columns to select.
     * @param column          The column to filter by.
     * @param data            The value to filter by.
     * @return A list of values from the selected columns, or an empty list if not found.
     */
    public List<Object> get(String table, String[] selectedColumns, String column, Object data) {
        String columns = String.join(", ", selectedColumns); // Join column names with commas
        String sql = "SELECT " + columns + " FROM " + table + " WHERE " + column + " = ?";
        List<Object> results = new ArrayList<>();

        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            pstmt.setObject(1, data);

            try (ResultSet res = pstmt.executeQuery()) {
                while (res.next()) {
                    Object row = null;
                    for (String selectedColumn : selectedColumns) {
                        row = res.getObject(selectedColumn); // Retrieve each column value
                        results.add(row);
                    }
                }
            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Database query failed: " + sql, e);
        }

        return results;
    }

    /**
     * Asynchronously retrieves multiple values from a specified table based on a column and data.
     *
     * @param table           The name of the table to query.
     * @param selectedColumns The columns to select.
     * @param column          The column to filter by.
     * @param data            The value to filter by.
     */
    public List<Object> get(String table, String[] selectedColumns, String[] column, Object[] data) {
        if (selectedColumns == null || selectedColumns.length == 0 ||
            column == null || data == null || column.length == 0 || data.length == 0 ||
            column.length != data.length) {
            throw new IllegalArgumentException("Selected columns, columns, and data arrays must not be empty and must have the same length.");
        }

        String columnsStr = String.join(", ", selectedColumns);

        // Build WHERE clause: col1 = ? AND col2 = ? ...
        StringBuilder whereClause = new StringBuilder();
        for (int i = 0; i < column.length; i++) {
            whereClause.append(column[i]).append(" = ?");
            if (i < column.length - 1) {
                whereClause.append(" AND ");
            }
        }

        String sql = "SELECT " + columnsStr + " FROM " + table + " WHERE " + whereClause;
        List<Object> results = new ArrayList<>();

        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            for (int i = 0; i < data.length; i++) {
                pstmt.setObject(i + 1, data[i]);
            }

            try (ResultSet res = pstmt.executeQuery()) {
                while (res.next()) {
                    for (String selectedColumn : selectedColumns) {
                        results.add(res.getObject(selectedColumn));
                    }
                }
            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Database query failed: " + sql, e);
        }

        return results;
    }

    /**
     * Asynchronously retrieves multiple values from a specified table based on a column and data.
     *
     * @param table           The name of the table to query.
     * @param selectedColumns The columns to select.
     * @param column          The column to filter by.
     * @param data            The value to filter by.
     * @param callback        The callback to handle the result.
     */
    public void getAsync(String table, String[] selectedColumns, String column, Object data, Callback<List<Object>> callback) {
        String columns = String.join(", ", selectedColumns); // Join column names with commas
        String sql = "SELECT " + columns + " FROM " + table + " WHERE " + column + " = ?";
        List<Object> results = new ArrayList<>();

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (PreparedStatement pstmt = result.prepareStatement(sql)) {

                        pstmt.setObject(1, data);

                        try (ResultSet res = pstmt.executeQuery()) {
                            while (res.next()) {
                                Object row = null;
                                for (String selectedColumn : selectedColumns) {
                                    row = res.getObject(selectedColumn); // Retrieve each column value
                                    results.add(row);
                                }
                            }
                            callback.onResult(results);
                        }
                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Retrieves a single value from a specified table based on a column and data, with type casting.
     *
     * @param table    The name of the table to query.
     * @param selected The column to select.
     * @param column   The column to filter by.
     * @param data     The value to filter by.
     * @param type     The expected type of the result.
     * @return The value from the selected column, cast to the specified type, or null if not found or type mismatch.
     */
    public <T> T get(String table, String selected, String column, Object data, Class<T> type) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            pstmt.setObject(1, data);

            try (ResultSet res = pstmt.executeQuery()) {
                if (res.next()) {
                    Object value = res.getObject(selected);
                    if (type.isInstance(value)) {
                        return type.cast(value);
                    } else {
                        LOGGER.warning("Type mismatch: Expected " + type.getSimpleName() + " but got " +
                                       (value != null ? value.getClass().getSimpleName() : "null"));
                    }
                }
            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Database query failed: " + sql, e);
        }
        return null;
    }

    public <T> Optional<T> getOptional(String table, String selected, String column, Object data, Class<T> type) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";
        try (Connection conn = connect();
             PreparedStatement pstmt = conn.prepareStatement(sql)) {

            pstmt.setObject(1, data);

            try (ResultSet res = pstmt.executeQuery()) {
                if (res.next()) {
                    Object value = res.getObject(selected);
                    if (type.isInstance(value)) {
                        return Optional.of(type.cast(value));
                    } else {
                        LOGGER.warning("Type mismatch: Expected " + type.getSimpleName() + " but got " +
                                       (value != null ? value.getClass().getSimpleName() : "null"));
                    }
                }
            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Database query failed: " + sql, e);
        }
        return Optional.empty();
    }

    /**
     * Asynchronously retrieves a single value from a specified table based on a column and data, with type casting.
     *
     * @param table    The name of the table to query.
     * @param selected The column to select.
     * @param column   The column to filter by.
     * @param data     The value to filter by.
     * @param type     The expected type of the result.
     * @param callback The callback to handle the result.
     */
    public <T> void getAsync(String table, String selected, String column, Object data, Class<T> type, Callback<T> callback) {
        String sql = "SELECT " + selected + " FROM " + table + " WHERE " + column + " = ?";

        executor.execute(() -> {
            connectAsync(new Callback<Connection>() {
                @Override
                public void onResult(Connection result) {
                    try (PreparedStatement pstmt = result.prepareStatement(sql)) {

                        pstmt.setObject(1, data);

                        try (ResultSet res = pstmt.executeQuery()) {
                            if (res.next()) {
                                Object value = res.getObject(selected);
                                if (type.isInstance(value)) {
                                    callback.onResult(type.cast(value));
                                } else {
                                    LOGGER.warning("Type mismatch: Expected " + type.getSimpleName() + " but got " +
                                                   (value != null ? value.getClass().getSimpleName() : "null"));
                                }
                            }
                        }
                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                }

                @Override
                public void onError(Exception e) {
                    callback.onError(e);
                }
            });
        });
    }

    /**
     * Adds a new column to an existing table.
     *
     * @param table  The name of the table to add the column to.
     * @param column The column definition (e.g., "new_column TEXT").
     * @return true if the column was added successfully, false otherwise.
     */
    public boolean addColumn(String table, String column) {
        String sql = "ALTER TABLE " + table + " ADD COLUMN " + column;
        if (!isTableExists(table))
            return false;
        try (Connection conn = connect();
             PreparedStatement statement = Objects.requireNonNull(conn).prepareStatement(sql)) {
            return statement.executeUpdate() > 0;

        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to add column: " + column + " to table: " + table, e);
        }
        return false;
    }

    /**
     * Asynchronously adds a new column to an existing table.
     *
     * @param table    The name of the table to add the column to.
     * @param column   The column definition (e.g., "new_column TEXT").
     * @param callback The callback to handle the result.
     */
    public void addColumnAsync(String table, String column, Callback<Boolean> callback) {
        String sql = "ALTER TABLE " + table + " ADD COLUMN " + column;

        isTableExistsAsync(table, new Callback<Boolean>() {
            @Override
            public void onResult(Boolean result) {
                if (!result) {
                    callback.onResult(false);
                    return;
                }
                connectAsync(new Callback<Connection>() {
                    @Override
                    public void onResult(Connection result) {
                        try (Connection conn = result;
                             PreparedStatement statement = Objects.requireNonNull(conn).prepareStatement(sql)) {

                            callback.onResult(statement.executeUpdate() > 0);

                        } catch (SQLException e) {
                            callback.onError(e);
                        }
                    }

                    @Override
                    public void onError(Exception e) {
                        callback.onError(e);
                    }
                });
            }

            @Override
            public void onError(Exception e) {
                callback.onError(e);
            }
        });
    }

    /**
     * Removes a column from an existing table.
     *
     * @param table  The name of the table to remove the column from.
     * @param column The name of the column to remove.
     * @return true if the column was removed successfully, false otherwise.
     */
    public boolean removeColumn(String table, String column) {
        if (!isTableExists(table))
            return false;
        try (Connection conn = connect()) {
            conn.setAutoCommit(false);

            // Step 1: Get the table's current structure
            String getColumnsSql = "PRAGMA table_info(" + table + ")";
            StringBuilder columns = new StringBuilder();
            try (PreparedStatement getColumnsStmt = conn.prepareStatement(getColumnsSql);
                 ResultSet rs = getColumnsStmt.executeQuery()) {

                while (rs.next()) {
                    String colName = rs.getString("name");
                    if (!colName.equalsIgnoreCase(column)) {
                        if (columns.length() > 0) {
                            columns.append(", ");
                        }
                        columns.append(colName);
                    }
                }
                if (columns.length() == 0) {
                    return false;
                }
                String columnsStr = columns.toString();

                // Step 2: Rename the old table
                String renameSql = "ALTER TABLE " + table + " RENAME TO " + table + "_old";
                try (PreparedStatement renameStmt = conn.prepareStatement(renameSql)) {
                    renameStmt.executeUpdate();
                }

                // Step 3: Create a new table without the column to be removed
                String createSql = "CREATE TABLE " + table + " AS SELECT " + columnsStr + " FROM " + table + "_old";
                try (PreparedStatement createStmt = conn.prepareStatement(createSql)) {
                    createStmt.executeUpdate();
                }

                // Step 4: Drop the old table
                String dropSql = "DROP TABLE " + table + "_old";
                try (PreparedStatement dropStmt = conn.prepareStatement(dropSql)) {
                    dropStmt.executeUpdate();
                }

                conn.commit();

            }
        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to remove column: " + column + " from table: " + table, e);
        }
        return true;
    }

    /**
     * Asynchronously removes a column from an existing table.
     *
     * @param table    The name of the table to remove the column from.
     * @param column   The name of the column to remove.
     * @param callback The callback to handle the result.
     */
    public void removeColumnAsync(String table, String column, Callback<Boolean> callback) {

        isTableExistsAsync(table, new Callback<Boolean>() {
            @Override
            public void onResult(Boolean result) {
                if (!result) {
                    callback.onResult(false);
                    return;
                }

                connectAsync(new Callback<Connection>() {
                    @Override
                    public void onResult(Connection result) {
                        try (Connection conn = result) {
                            conn.setAutoCommit(false);

                            // Step 1: Get the table's current structure
                            String getColumnsSql = "PRAGMA table_info(" + table + ")";
                            StringBuilder columns = new StringBuilder();
                            try (PreparedStatement getColumnsStmt = conn.prepareStatement(getColumnsSql);
                                 ResultSet rs = getColumnsStmt.executeQuery()) {

                                while (rs.next()) {
                                    String colName = rs.getString("name");
                                    if (!colName.equalsIgnoreCase(column)) {
                                        if (columns.length() > 0) {
                                            columns.append(", ");
                                        }
                                        columns.append(colName);
                                    }
                                }
                                if (columns.length() == 0) {
                                    callback.onResult(false);
                                    return;
                                }
                                String columnsStr = columns.toString();

                                // Step 2: Rename the old table
                                String renameSql = "ALTER TABLE " + table + " RENAME TO " + table + "_old";
                                try (PreparedStatement renameStmt = conn.prepareStatement(renameSql)) {
                                    renameStmt.executeUpdate();
                                }

                                // Step 3: Create a new table without the column to be removed
                                String createSql = "CREATE TABLE " + table + " AS SELECT " + columnsStr + " FROM " + table + "_old";
                                try (PreparedStatement createStmt = conn.prepareStatement(createSql)) {
                                    createStmt.executeUpdate();
                                }

                                // Step 4: Drop the old table
                                String dropSql = "DROP TABLE " + table + "_old";
                                try (PreparedStatement dropStmt = conn.prepareStatement(dropSql)) {
                                    dropStmt.executeUpdate();
                                }

                                conn.commit();

                            }
                        } catch (SQLException e) {
                            callback.onError(e);
                        }
                    }

                    @Override
                    public void onError(Exception e) {
                        callback.onError(e);
                    }
                });

            }

            @Override
            public void onError(Exception e) {
                callback.onError(e);
            }
        });
    }

    /**
     * Retrieves the names of all columns in a specified table.
     *
     * @param table The name of the table to retrieve columns from.
     * @return A list of column names in the specified table.
     */
    public List<String> showColumns(String table) {
        List<String> columns = new ArrayList<>();
        String sql = "SELECT * FROM " + table;
        try (Connection conn = connect();
             Statement statement = Objects.requireNonNull(conn).createStatement();
             ResultSet rs = statement.executeQuery(sql)) {

            ResultSetMetaData mrs = rs.getMetaData();
            for (int i = 1; i <= mrs.getColumnCount(); i++) {
                columns.add(mrs.getColumnLabel(i));
            }
            return columns;

        } catch (SQLException e) {
            LOGGER.log(Level.SEVERE, "Failed to retrieve columns from table: " + table, e);
        }
        return columns;
    }

    /**
     * Asynchronously retrieves the names of all columns in a specified table.
     *
     * @param table    The name of the table to retrieve columns from.
     * @param callback The callback to handle the result.
     */
    public void showColumnsAsync(String table, Callback<List<String>> callback) {
        List<String> columns = new ArrayList<>();
        String sql = "SELECT * FROM " + table;

        connectAsync(new Callback<Connection>() {
            @Override
            public void onResult(Connection result) {
                executor.execute(() -> {
                    try (Connection conn = result;
                         Statement statement = Objects.requireNonNull(conn).createStatement();
                         ResultSet rs = statement.executeQuery(sql)) {

                        ResultSetMetaData mrs = rs.getMetaData();
                        for (int i = 1; i <= mrs.getColumnCount(); i++) {
                            columns.add(mrs.getColumnLabel(i));
                        }
                        callback.onResult(columns);

                    } catch (SQLException e) {
                        callback.onError(e);
                    }
                });
            }

            @Override
            public void onError(Exception e) {
                callback.onError(e);
            }
        });
    }
}
