package io.github.kaststream.api.v2;

import io.github.kaststream.api.v2.facade.KastResultFuture;
import io.github.kaststream.api.v2.facade.KastRow;
import io.github.kaststream.api.v2.facade.KastRowBuilder;
import java.io.Serializable;
import java.util.Map;
import org.apache.avro.Schema;

public interface AsyncProcessor extends Serializable {
  /**
   * Declare the data type information of the data produced by your function.
   *
   * @param schema the input row schema
   * @return the output row type information
   */
  public Schema getProducedType(final Schema schema);

  /**
   * Initialization method for the function. It is called before the actual working method.
   *
   * @param configuration The configuration containing the parameters attached to the contract.
   * @param schema the input row schema
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(final Map<String, String> configuration, final Schema schema) throws Exception {};

  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema the input row schema
   * @param meta the udf configuration parameters
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(
      final Map<String, String> configuration, final Schema schema, Map<String, Object> meta)
      throws Exception {
    throw new UnsupportedOperationException("open with meta not implemented");
  }
  ;

  /**
   * Tear-down method for the user code (ex: clean resources from open).
   *
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void close() throws Exception {};

  /**
   * Trigger async operation for each stream input.
   *
   * @param input element coming from an upstream task
   * @param resultFuture to be completed with the result data
   * @throws Exception in case of a user code error. An exception will make the task fail and
   *     trigger fail-over process.
   */
  public void asyncInvoke(
      KastRow input, KastResultFuture resultFuture, KastRowBuilder kastRowBuilder) throws Exception;
}
