package io.github.kaststream.api.v2;

import io.github.kaststream.api.v2.facade.KastRow;
import java.io.Serializable;
import java.util.Map;
import org.apache.avro.Schema;

/** Implement this interface to provide your own sink function. */
public interface KastSink extends Serializable {
  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(Map<String, String> configuration, Schema schema) throws Exception {};
  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema the input row schema
   * @param meta the udf configuration parameters
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(
      final Map<String, String> configuration, final Schema schema, Map<String, Object> meta)
      throws Exception {
    throw new UnsupportedOperationException("open with meta not implemented");
  }
  ;
  /**
   * Trigger operation for each stream input.
   *
   * @param row row element coming from an upstream task
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public void invoke(KastRow row) throws Exception;

  /**
   * Tear-down method for the user code (ex: clean resources from open).
   *
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void close() throws Exception {};

  /**
   * This method is called when a snapshot for a checkpoint is requested.
   *
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public void snapshotState() throws Exception;

  /**
   * This method is called when the parallel function instance is created during distributed
   * execution. Functions typically set up their state storing data structures in this method.
   *
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public void initializeState() throws Exception;
}
