package io.github.kaststream.api.v2;

import io.github.kaststream.api.v2.facade.KastRow;
import io.github.kaststream.api.v2.facade.KastRowBuilder;
import java.io.Serializable;
import java.util.Map;
import org.apache.avro.Schema;

/** The map function. */
public interface Mapper extends Serializable {

  /**
   * Your function implementation. Do what you need on the received rows.
   *
   * @param kRow the received row
   * @return the transformed row
   * @throws Exception in case of problem. This will stop your dag.
   */
  public KastRow map(final KastRow kRow, KastRowBuilder kastRowBuilder) throws Exception;

  /**
   * Declare the data type information of the data produced by your function.
   *
   * @param schema the input row schema
   * @return the produced data typeInfo
   */
  public Schema getProducedType(final Schema schema);

  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema the input row schema
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(final Map<String, String> configuration, final Schema schema)
      throws Exception {}
  ;

  /**
   * Initialization method for the function.
   *
   * @param configuration The configuration containing the parameters attached to the contract
   * @param schema the input row schema
   * @param meta the udf configuration parameters
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void open(
      final Map<String, String> configuration, final Schema schema, Map<String, Object> meta)
      throws Exception {
    throw new UnsupportedOperationException("open with meta not implemented");
  }
  ;

  /**
   * Tear-down method for the user code. It is called after the last call to the main working
   * methods. For functions that are part of an iteration, this method will be invoked after each
   * iteration superstep.
   *
   * <p>This method can be used for clean up work.
   *
   * @throws Exception The function may throw exceptions which cause the streaming program to fail
   *     and go into recovery.
   */
  public default void close() throws Exception {}
}
