package io.github.kaststream.api.v2.facade;

public interface KastRuntimeContext {

  /**
   * The ID of the current job. Note that Job ID can change in particular upon manual restart. The
   * returned ID should NOT be used for any job management tasks.
   */
  String getJobId();

  /**
   * Returns the name of the task in which the UDF runs, as assigned during plan construction.
   *
   * @return The name of the task in which the UDF runs.
   */
  String getTaskName();

  /**
   * Gets the parallelism with which the parallel task runs.
   *
   * @return The parallelism with which the parallel task runs.
   */
  int getNumberOfParallelSubtasks();

  /**
   * Gets the number of max-parallelism with which the parallel task runs.
   *
   * @return The max-parallelism with which the parallel task runs.
   */
  int getMaxNumberOfParallelSubtasks();

  /**
   * Gets the number of this parallel subtask. The numbering starts from 0 and goes up to
   * parallelism-1 (parallelism as returned by {@link #getNumberOfParallelSubtasks()}).
   *
   * @return The index of the parallel subtask.
   */
  int getIndexOfThisSubtask();

  /**
   * Gets the attempt number of this parallel subtask. First attempt is numbered 0.
   *
   * @return Attempt number of the subtask.
   */
  int getAttemptNumber();

  /**
   * Returns the name of the task, appended with the subtask indicator, such as "MyTask (3/6)#1",
   * where 3 would be ({@link #getIndexOfThisSubtask()} + 1), and 6 would be {@link
   * #getNumberOfParallelSubtasks()}, and 1 would be {@link #getAttemptNumber()}.
   *
   * @return The name of the task, with subtask indicator.
   */
  String getTaskNameWithSubtasks();

  /**
   * Gets the ClassLoader to load classes that are not in system's classpath, but are part of the
   * jar file of a user job.
   *
   * @return The ClassLoader for user code classes.
   */
  ClassLoader getUserCodeClassLoader();

  /**
   * Tests for the existence of the broadcast variable identified by the given {@code name}.
   *
   * @param name The name under which the broadcast variable is registered;
   * @return Whether a broadcast variable exists for the given name.
   */
  boolean hasBroadcastVariable(String name);

  /**
   * Gets a handle to the system's key/value map state.
   *
   * <p>This state is only accessible if the function is executed on a KeyedStream.
   *
   * <pre>{@code
   * DataStream<MyType> stream = ...;
   * KeyedStream<MyType> keyedStream = stream.keyBy("id");
   *
   * keyedStream.map(new RichMapFunction<MyType, List<MyType>>() {
   *
   *     private MapState<MyType, Long> state;
   *
   *     public void open(Configuration cfg) {
   *         state = getRuntimeContext().getMapState(
   *                 new MapStateDescriptor<>("sum", MyType.class, Long.class));
   *     }
   *
   *     public Tuple2<MyType, Long> map(MyType value) {
   *         return new Tuple2<>(value, state.get(value));
   *     }
   * });
   *
   * }</pre>
   *
   * @param <UK> The type of the user keys stored in the state.
   * @param <UV> The type of the user values stored in the state.
   * @return The partitioned state object.
   * @throws UnsupportedOperationException Thrown, if no partitioned state is available for the
   *     function (function is not part of a KeyedStream).
   */
  <UK, UV> KastMapState<UK, UV> getMapState(String name, Class<UK> keyClass, Class<UV> valueClass);
}
