/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.drill.exec.store;

import org.apache.drill.shaded.guava.com.google.common.collect.Lists;
import org.apache.drill.shaded.guava.com.google.common.collect.Maps;

import org.apache.drill.common.expression.SchemaPath;
import org.apache.drill.exec.record.VectorAccessible;
import org.apache.drill.exec.record.VectorWrapper;
import org.apache.drill.exec.vector.complex.impl.UnionReader;
import org.apache.drill.exec.vector.complex.reader.FieldReader;

import java.io.IOException;
import java.util.List;
import java.util.Map;

/*
 * This class is generated using freemarker and the EventBasedRecordWriter.java template.
 */

/** Reads records from the RecordValueAccessor and writes into RecordWriter. */
public class EventBasedRecordWriter {
  static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(EventBasedRecordWriter.class);

  private VectorAccessible batch;
  private RecordWriter recordWriter;
  private List<FieldConverter> fieldConverters;

  public EventBasedRecordWriter(VectorAccessible batch, RecordWriter recordWriter)
          throws IOException {
    this.batch = batch;
    this.recordWriter = recordWriter;

    initFieldWriters();
  }

  public int write(int recordCount) throws IOException {
    int counter = 0;

    for (; counter < recordCount; counter++) {
      recordWriter.checkForNewPartition(counter);
      recordWriter.startRecord();
      // write the current record
      for (FieldConverter converter : fieldConverters) {
        converter.setPosition(counter);
        converter.startField();
        converter.writeField();
        converter.endField();
      }
      recordWriter.endRecord();
    }

    return counter;
  }

  private void initFieldWriters() throws IOException {
    fieldConverters = Lists.newArrayList();
    try {
      int fieldId = 0;
      for (VectorWrapper w : batch) {
        if (!recordWriter.supportsField(w.getField())) {
          continue;
        }
        FieldReader reader = w.getValueVector().getReader();
        FieldConverter converter = getConverter(recordWriter, fieldId++, w.getField().getName(), reader);
        fieldConverters.add(converter);
      }
    } catch(Exception e) {
      logger.error("Failed to create FieldWriter.", e);
      throw new IOException("Failed to initialize FieldWriters.", e);
    }
  }

  public static abstract class FieldConverter {
    protected int fieldId;
    protected String fieldName;
    protected FieldReader reader;

    public FieldConverter(int fieldId, String fieldName, FieldReader reader) {
      this.fieldId = fieldId;
      this.fieldName = fieldName;
      this.reader = reader;
    }

    public void setPosition(int index) {
      reader.setPosition(index);
    }

    public void startField() throws IOException {
      // no op
    }

    public void endField() throws IOException {
      // no op
    }

    public abstract void writeField() throws IOException;

    /**
     * Used by repeated converters for writing Parquet logical lists.
     *
     * @throws IOException may be thrown by subsequent invocation of {{@link #writeField()}}
     *         in overriden methods
     * @see <a href="https://github.com/apache/parquet-format/blob/master/LogicalTypes.md#lists">Lists</a>
     */
    public void writeListField() throws IOException {
      throw new UnsupportedOperationException(String.format(
          "Converter '%s' doesn't support writing list fields.",
          getClass().getSimpleName()));
    }
  }

  public static FieldConverter getConverter(RecordWriter recordWriter, int fieldId, String fieldName, FieldReader reader) {
    switch (reader.getType().getMinorType()) {
      case UNION:
        return recordWriter.getNewUnionConverter(fieldId, fieldName, reader);
      case MAP:
        switch (reader.getType().getMode()) {
          case REQUIRED:
          case OPTIONAL:
            return recordWriter.getNewMapConverter(fieldId, fieldName, reader);
          case REPEATED:
            return recordWriter.getNewRepeatedMapConverter(fieldId, fieldName, reader);
        }

      case DICT:
        switch (reader.getType().getMode()) {
          case REQUIRED:
          case OPTIONAL:
            return recordWriter.getNewDictConverter(fieldId, fieldName, reader);
          case REPEATED:
            return recordWriter.getNewRepeatedDictConverter(fieldId, fieldName, reader);
        }

      case LIST:
        return recordWriter.getNewRepeatedListConverter(fieldId, fieldName, reader);

      case NULL:
        return recordWriter.getNewNullableIntConverter(fieldId, fieldName, reader);

      case TINYINT:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewTinyIntConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableTinyIntConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedTinyIntConverter(fieldId, fieldName, reader);
      }
      case UINT1:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewUInt1Converter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableUInt1Converter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedUInt1Converter(fieldId, fieldName, reader);
      }
      case UINT2:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewUInt2Converter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableUInt2Converter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedUInt2Converter(fieldId, fieldName, reader);
      }
      case SMALLINT:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewSmallIntConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableSmallIntConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedSmallIntConverter(fieldId, fieldName, reader);
      }
      case INT:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewIntConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableIntConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedIntConverter(fieldId, fieldName, reader);
      }
      case UINT4:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewUInt4Converter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableUInt4Converter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedUInt4Converter(fieldId, fieldName, reader);
      }
      case FLOAT4:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewFloat4Converter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableFloat4Converter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedFloat4Converter(fieldId, fieldName, reader);
      }
      case TIME:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewTimeConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableTimeConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedTimeConverter(fieldId, fieldName, reader);
      }
      case INTERVALYEAR:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewIntervalYearConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableIntervalYearConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedIntervalYearConverter(fieldId, fieldName, reader);
      }
      case DECIMAL9:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewDecimal9Converter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableDecimal9Converter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedDecimal9Converter(fieldId, fieldName, reader);
      }
      case BIGINT:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewBigIntConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableBigIntConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedBigIntConverter(fieldId, fieldName, reader);
      }
      case UINT8:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewUInt8Converter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableUInt8Converter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedUInt8Converter(fieldId, fieldName, reader);
      }
      case FLOAT8:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewFloat8Converter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableFloat8Converter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedFloat8Converter(fieldId, fieldName, reader);
      }
      case DATE:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewDateConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableDateConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedDateConverter(fieldId, fieldName, reader);
      }
      case TIMESTAMP:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewTimeStampConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableTimeStampConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedTimeStampConverter(fieldId, fieldName, reader);
      }
      case DECIMAL18:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewDecimal18Converter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableDecimal18Converter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedDecimal18Converter(fieldId, fieldName, reader);
      }
      case INTERVALDAY:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewIntervalDayConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableIntervalDayConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedIntervalDayConverter(fieldId, fieldName, reader);
      }
      case INTERVAL:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewIntervalConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableIntervalConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedIntervalConverter(fieldId, fieldName, reader);
      }
      case DECIMAL28DENSE:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewDecimal28DenseConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableDecimal28DenseConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedDecimal28DenseConverter(fieldId, fieldName, reader);
      }
      case DECIMAL38DENSE:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewDecimal38DenseConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableDecimal38DenseConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedDecimal38DenseConverter(fieldId, fieldName, reader);
      }
      case DECIMAL38SPARSE:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewDecimal38SparseConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableDecimal38SparseConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedDecimal38SparseConverter(fieldId, fieldName, reader);
      }
      case DECIMAL28SPARSE:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewDecimal28SparseConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableDecimal28SparseConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedDecimal28SparseConverter(fieldId, fieldName, reader);
      }
      case VARBINARY:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewVarBinaryConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableVarBinaryConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedVarBinaryConverter(fieldId, fieldName, reader);
      }
      case VARCHAR:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewVarCharConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableVarCharConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedVarCharConverter(fieldId, fieldName, reader);
      }
      case VAR16CHAR:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewVar16CharConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableVar16CharConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedVar16CharConverter(fieldId, fieldName, reader);
      }
      case VARDECIMAL:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewVarDecimalConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableVarDecimalConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedVarDecimalConverter(fieldId, fieldName, reader);
      }
      case BIT:
      switch (reader.getType().getMode()) {
        case REQUIRED:
          return recordWriter.getNewBitConverter(fieldId, fieldName, reader);
        case OPTIONAL:
          return recordWriter.getNewNullableBitConverter(fieldId, fieldName, reader);
        case REPEATED:
          return recordWriter.getNewRepeatedBitConverter(fieldId, fieldName, reader);
      }

    }
    throw new UnsupportedOperationException();
  }
}
