// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.List;

/**
 * The {@link ArgsSyntax} interface represents a node (and therewith the syntax
 * for the arguments) in the command line arguments syntax tree; simply
 * extending the {@link Syntaxable} interface and adding the functionality of
 * providing access to the added {@link Operand}s (leafs). In future extensions,
 * a {@link ArgsSyntax} might provide access to the child {@link Syntaxable}
 * elements contained in a {@link ArgsSyntax} instance. As of the current
 * findings, access to the children of the {@link ArgsSyntax} node is not
 * required and would make the interface unnecessarily complicated.
 */
public interface ArgsSyntax extends Syntaxable {

	/**
	 * Traverses the hierarchy of {@link ArgsSyntax}s and collects all therein
	 * found {@link Operand}s and sub-types such as the {@link Flag} or the
	 * {@link Option}. This is most useful when creating a list of
	 * {@link Option}s, {@link Flag}es and {@link Operand}s when printing out
	 * detailed help.
	 * 
	 * @return The according {@link Operand} elements.
	 */
	List<? extends Operand<?>> toOperands();

	/**
	 * Traverses the hierarchy of {@link ArgsSyntax} tree and returns the
	 * (first) operand matching the given alias.
	 * 
	 * @param aAlias The alias for which to seek for.
	 * 
	 * @return The according {@link Operand} element or null if none matching
	 *         was found.
	 */
	default Operand<?> toOperand( String aAlias ) {
		return toOperand( aAlias, null );
	}

	/**
	 * Traverses the hierarchy of {@link ArgsSyntax} tree and returns the
	 * (first) operand matching the given type.
	 *
	 * @param <T> the generic type
	 * @param aType The type for which to seek for.
	 * 
	 * @return The according {@link Operand} element or null if none matching
	 *         was found
	 */
	default <T extends Operand<?>> T toOperand( Class<T> aType ) {
		return toOperand( null, aType );
	}

	/**
	 * Traverses the hierarchy of {@link ArgsSyntax} tree and returns the
	 * (first) operand matching the given alias and type.
	 *
	 * @param <T> the generic type
	 * @param aAlias The alias for which to seek for.
	 * @param aType The type for which to seek for.
	 * 
	 * @return The according {@link Operand} element or null if none matching
	 *         was found
	 */
	<T extends Operand<?>> T toOperand( String aAlias, Class<T> aType );

}
