// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

import org.refcodes.mixin.EnabledAccessor;

/**
 * The {@link Flag} class implements the {@link Option} interface for
 * representing either a <code>true</code> or a <code>false</code> state: When a
 * flag is provided to your command line arguments, then it is considered to be
 * <code>true</code>, when it is omitted, then it is considered to be
 * <code>false</code> as of {@link #isEnabled()}.
 */
public class Flag extends AbstractOption<Boolean> implements Option<Boolean>, EnabledAccessor {

	/**
	 * Constructs a {@link Flag}.
	 *
	 * @param aShortOption The short-option being a single character with the
	 *        additional single hyphen-minus "-" prefix.
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aAlias The alias identifying this {@link Flag}.
	 * @param aDescription A description without any line breaks.
	 */
	public Flag( String aShortOption, String aLongOption, String aAlias, String aDescription ) {
		super( aShortOption, aLongOption, Boolean.class, aAlias, aDescription );
		setValue( false );
	}

	/**
	 * Constructs a {@link Flag}.
	 *
	 * @param aLongOption The long-option being a multi-character sequence with
	 *        at least two characters with the additional double hyphen-minus
	 *        "--" prefix.
	 * @param aAlias The alias identifying this {@link Flag}.
	 * @param aDescription A description without any line breaks.
	 */
	public Flag( String aLongOption, String aAlias, String aDescription ) {
		this( null, aLongOption, aAlias, aDescription );
		setValue( false );
	}

	/**
	 * Constructs a {@link Flag}.
	 *
	 * @param aAlias The alias identifying this {@link Flag}.
	 * @param aValue The value to be pre-set for the {@link Flag}.
	 */
	public Flag( String aAlias, boolean aValue ) {
		super( null, null, Boolean.class, aAlias, null );
		setValue( aValue );
	}

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Returns true if the switch has been set (enabled).
	 * 
	 * @return True in case the {@link Flag} has been provided (set), else
	 *         false.
	 */
	@Override
	public boolean isEnabled() {
		Boolean theValue = getValue();
		if ( theValue != null ) {
			return theValue;
		}
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Operand<Boolean>> parseArgs( String[] aArgs, String[] aOptions ) throws ArgsSyntaxException {
		for ( String eArg : aArgs ) {
			String theOption = getShortOption();
			if ( eArg.equals( theOption ) ) {
				List<Operand<Boolean>> theList = new ArrayList<Operand<Boolean>>();
				theList.add( this );
				setValue( true );
				setArgs( new String[] {
						theOption
				} );
				return theList;
			}
			theOption = getLongOption();
			if ( eArg.equals( theOption ) ) {
				List<Operand<Boolean>> theList = new ArrayList<Operand<Boolean>>();
				theList.add( this );
				setValue( true );
				setArgs( new String[] {
						theOption
				} );
				return theList;
			}
		}
		throw new UnknownArgsException( aArgs, "Neither the short-option \"" + getShortOption() + "\" nor the long-option \"" + getLongOption() + "\"  was set in the command line arguments, at least one of them must be set." );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		aOptEscCode = aOptEscCode == null ? "" : aOptEscCode;
		aParamEscCode = aParamEscCode == null ? "" : aParamEscCode;
		aResetEscCode = aResetEscCode == null ? "" : aResetEscCode;
		return aOptEscCode + (getShortOption() != null ? getShortOption() : getLongOption()) + aResetEscCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void reset() {
		super.reset();
		super.setValue( false );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Boolean toType( String aArg ) throws ParseArgsException {
		throw new UnsupportedOperationException( "*** NOT REQUIRED BY THIS IMPLEMENTATION ***" );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
