// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.ArrayList;
import java.util.List;

/**
 * An {@link OrCondition} represents a list of {@link ArgsSyntax}
 * ({@link Syntaxable}) instances of which at least one must be parsed
 * successfully when the {@link Syntaxable} s'
 * {@link Syntaxable#parseArgs(String[], String[])} methods are invoked. The
 * command line arguments syntax <code>{ -a | -b | -c }</code> specifies that at
 * least "-a", "-b" or "-c" must be set. In case none is set , then the
 * {@link OrCondition} will terminate the {@link #parseArgs(String[], String[])}
 * method with an exception.
 */
public class OrCondition extends AbstractCondition implements Condition {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String RIGHT_TAG = " )";
	private static final String LEFT_TAG = "( ";
	private static final String GNU_POSIX_OR = " ? ";
	private static final String LOGICAL_OR = " | ";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link OrCondition} with the {@link Syntaxable}
	 * ({@link ArgsSyntax}) instances to be nested.
	 *
	 * @param aArgs The {@link Syntaxable} ({@link ArgsSyntax}) instances to be
	 *        nested.
	 */
	public OrCondition( Syntaxable... aArgs ) {
		super( aArgs );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions ) throws ArgsSyntaxException {
		List<Operand<?>> theResult = null;
		List<Operand<?>> eOperands;
		for ( Syntaxable eChild : getChildren() ) {
			try {
				eOperands = (List<Operand<?>>) eChild.parseArgs( aArgs, aOptions );
				if ( eOperands != null ) {
					if ( theResult == null ) {
						theResult = new ArrayList<Operand<?>>();
						;
					}
					theResult.addAll( eOperands );
				}
				aArgs = CliUtility.toDiff( aArgs, eOperands );
			}
			catch ( UnknownArgsException | AmbiguousArgsException exc ) {}
		}
		if ( theResult != null ) {
			return theResult;
		}
		throw new UnknownArgsException( aArgs, "Not any syntax matched the provided command line arguments, though at least one elective must match." );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eChild : getChildren() ) {
			if ( theBuilder.length() == 0 ) {
				theBuilder.append( LEFT_TAG );
			}
			else {
				if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
					theBuilder.append( LOGICAL_OR );
				}
				else {
					theBuilder.append( GNU_POSIX_OR );
				}
			}
			theBuilder.append( eChild.toSyntax( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode ) );
		}
		if ( theBuilder.length() != 0 ) {
			theBuilder.append( RIGHT_TAG );
		}
		return theBuilder.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSynopsis( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eChild : getChildren() ) {
			if ( theBuilder.length() != 0 ) {
				if ( aSyntaxNotation == SyntaxNotation.REFCODES ) {
					theBuilder.append( LOGICAL_OR );
				}
				else {
					theBuilder.append( GNU_POSIX_OR );
				}
			}
			theBuilder.append( eChild.toSyntax( aSyntaxNotation, aOptEscCode, aParamEscCode, aResetEscCode ) );
		}
		return theBuilder.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toState() {
		StringBuilder theBuilder = new StringBuilder();
		for ( Syntaxable eChild : getChildren() ) {
			if ( theBuilder.length() == 0 ) {
				theBuilder.append( LEFT_TAG );
			}
			else {
				theBuilder.append( LOGICAL_OR );
			}
			theBuilder.append( eChild.toState() );
		}
		if ( theBuilder.length() != 0 ) {
			theBuilder.append( RIGHT_TAG );
		}
		return theBuilder.toString();

	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
