// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.cli;

import java.util.List;

import org.refcodes.component.Resetable;

/**
 * A {@link Syntaxable} defines the methods at least required when building a
 * command line arguments syntax tree for traversing the syntax tree; either for
 * parsing command line arguments or for constructing the command line arguments
 * syntax. By providing various implementations of the {@link Syntaxable}'s
 * subclasses such as {@link Operand}, {@link Option} or {@link ArgsSyntax}, a
 * command line arguments syntax tree can be constructed. This syntax tree can
 * be use to create a human readable (verbose) command line arguments syntax and
 * to parse an array of command line arguments for determining the
 * {@link Operand}s', the {@link Flag}es' or the {@link Option}s' values.
 */
public interface Syntaxable extends Resetable, Synopsis {

	/**
	 * This method is to be called from inside the {@link Syntaxable} hierarchy;
	 * use the method {@link #toSynopsis(SyntaxNotation)} in case you invoke
	 * syntax retrieval from the root {@link Syntaxable}. Returns the human
	 * readable (verbose) syntax of this {@link Syntaxable} instance including,
	 * in case of being a node in the syntax tree (such as a {@link ArgsSyntax},
	 * the syntax of the child {@link Syntaxable} instances. ATTENTION: As of
	 * different parenthesis settings for some notations regarding the root
	 * {@link Syntaxable} and the child {@link Syntaxable}s, the method
	 * {@link #toSyntax(SyntaxNotation, String, String, String)} is called from
	 * inside the {@link Syntaxable} hierarchy. In case the syntax is to be
	 * retrieved from the root {@link Syntaxable}, then the applicable method to
	 * be called is {@link #toSynopsis(SyntaxNotation)}, as for some notations
	 * it will for example not create the most outer braces.
	 * 
	 * @param aSyntaxNotation The syntax notation used for generating the
	 *        command line arguments syntax.
	 * 
	 * @return The human readable (verbose) command line arguments syntax.
	 */
	default String toSyntax( SyntaxNotation aSyntaxNotation ) {
		return toSyntax( aSyntaxNotation, null, null, null );
	}

	/**
	 * This method is to be called from inside the {@link Syntaxable} hierarchy;
	 * use the method {@link #toSynopsis(SyntaxNotation)} in case you invoke
	 * syntax retrieval from the root {@link Syntaxable}. Returns the human
	 * readable (verbose) syntax of this {@link Syntaxable} instance including,
	 * in case of being a node in the syntax tree (such as a {@link ArgsSyntax},
	 * the syntax of the child {@link Syntaxable} instances. ATTENTION: As of
	 * different parenthesis settings for some notations regarding the root
	 * {@link Syntaxable} and the child {@link Syntaxable}s, the method
	 * {@link #toSyntax(SyntaxNotation, String, String, String)} is called from
	 * inside the {@link Syntaxable} hierarchy. In case the syntax is to be
	 * retrieved from the root {@link Syntaxable}, then the applicable method to
	 * be called is {@link #toSynopsis(SyntaxNotation)}, as for some notations
	 * it will for example not create the most outer braces.
	 * 
	 * This method uses the {@link SyntaxNotation#REFCODES} be default!
	 * 
	 * @return The human readable (verbose) command line arguments syntax.
	 */
	default String toSyntax() {
		return toSyntax( SyntaxNotation.REFCODES, null, null, null );
	}

	/**
	 * This method is to be called from inside the {@link Syntaxable} hierarchy;
	 * use the method {@link #toSynopsis(SyntaxNotation)} in case you invoke
	 * syntax retrieval from the root {@link Syntaxable}. Returns the human
	 * readable (verbose) syntax of this {@link Syntaxable} instance including,
	 * in case of being a node in the syntax tree (such as a {@link ArgsSyntax},
	 * the syntax of the child {@link Syntaxable} instances. ATTENTION: As of
	 * different parenthesis settings for some notations regarding the root
	 * {@link Syntaxable} and the child {@link Syntaxable}s, the method
	 * {@link #toSyntax(SyntaxNotation, String, String, String)} is called from
	 * inside the {@link Syntaxable} hierarchy. In case the syntax is to be
	 * retrieved from the root {@link Syntaxable}, then the applicable method to
	 * be called is {@link #toSynopsis(SyntaxNotation)}, as for some notations
	 * it will for example not create the most outer braces.
	 * 
	 * This method uses the {@link SyntaxNotation#REFCODES} be default!
	 * 
	 * @param aOptEscCode The escape code to be used when processing an option,
	 *        e.g. this can be an ANSI Escape-Code to highlight the option.
	 * @param aParamEscCode The escape code to be used when processing a
	 *        keyword, e.g. this can be an ANSI Escape-Code to highlight the
	 *        option and/or parameters.
	 * @param aResetEscCode The escape code to close (reset) any Escape-Code
	 *        being set before.
	 * 
	 * @return The human readable (verbose) command line arguments syntax.
	 */
	default String toSyntax( String aOptEscCode, String aParamEscCode, String aResetEscCode ) {
		return toSyntax( SyntaxNotation.REFCODES, aOptEscCode, aParamEscCode, aResetEscCode );
	}

	/**
	 * This method is to be called from inside the {@link Syntaxable} hierarchy;
	 * use the method {@link #toSynopsis(SyntaxNotation)} in case you invoke
	 * syntax retrieval from the root {@link Syntaxable}. Returns the human
	 * readable (verbose) syntax of this {@link Syntaxable} instance including,
	 * in case of being a node in the syntax tree (such as a {@link ArgsSyntax},
	 * the syntax of the child {@link Syntaxable} instances. ATTENTION: As of
	 * different parenthesis settings for some notations regarding the root
	 * {@link Syntaxable} and the child {@link Syntaxable}s, the method
	 * {@link #toSyntax(SyntaxNotation, String, String, String)} is called from
	 * inside the {@link Syntaxable} hierarchy. In case the syntax is to be
	 * retrieved from the root {@link Syntaxable}, then the applicable method to
	 * be called is {@link #toSynopsis(SyntaxNotation)}, as for some notations
	 * it will for example not create the most outer braces.
	 * 
	 * @param aSyntaxNotation The syntax notation used for generating the
	 *        command line arguments syntax.
	 * @param aOptEscCode The escape code to be used when processing an option,
	 *        e.g. this can be an ANSI Escape-Code to highlight the option.
	 * @param aParamEscCode The escape code to be used when processing a
	 *        keyword, e.g. this can be an ANSI Escape-Code to highlight the
	 *        option and/or parameters.
	 * @param aResetEscCode The escape code to close (reset) any Escape-Code
	 *        being set before.
	 * 
	 * @return The human readable (verbose) command line arguments syntax.
	 */
	String toSyntax( SyntaxNotation aSyntaxNotation, String aOptEscCode, String aParamEscCode, String aResetEscCode );

	/**
	 * Parses the provided command line arguments and determines the according
	 * values by evaluating this {@link Syntaxable} instance or, in case of
	 * being a node in the syntax tree (such as a {@link ArgsSyntax}, traversing
	 * the child {@link Syntaxable} instances'
	 * {@link #parseArgs(String[], String[])} methods. In case of successfully
	 * parsing this {@link Syntaxable} and / or the child {@link Syntaxable}
	 * instances, the evaluated command line arguments are returned: Depending
	 * on the {@link Syntaxable} subclasses representing the evaluated command
	 * line arguments, instances of {@link Flag} classes, {@link Option} classes
	 * or {@link Operand} classes may be found in the result. In case of parsing
	 * failure, an according exception is thrown. ATTENTION: This method does
	 * not test for superfluous command line arguments being passed; e.g.
	 * command line arguments not being evaluated by any of the
	 * {@link Syntaxable} instance being traversed. This method is commonly used
	 * by a root {@link Syntaxable}'s {@link ArgsParser#evalArgs(String[])}
	 * method, which delegates to the {@link #parseArgs(String[], String[])}
	 * method and after which it determines whether there are superfluous
	 * arguments to be taken care of (by throwing an according exception).
	 * Business logic therefore should invoke the root node's
	 * {@link #parseArgs(String[], String[])} method as ignoring superfluous
	 * command line arguments will cause unexpected behavior from the point of
	 * view of the invoker.
	 * 
	 * @param aArgs The command line arguments to be parsed.
	 * 
	 * @return The list of evaluated command line arguments being instances of
	 *         the {@link Operand} interfaces or its sub-types.
	 * 
	 * @throws ArgsSyntaxException thrown in case of a command line arguments
	 *         mismatch regarding provided and expected args.
	 */
	default List<? extends Operand<?>> parseArgs( String[] aArgs ) throws ArgsSyntaxException {
		String theOptions[] = CliUtility.toOptions( this );
		return parseArgs( aArgs, theOptions );
	}

	/**
	 * Parses the provided command line arguments and determines the according
	 * values by evaluating this {@link Syntaxable} instance or, in case of
	 * being a node in the syntax tree (such as a {@link ArgsSyntax}, traversing
	 * the child {@link Syntaxable} instances'
	 * {@link #parseArgs(String[], String[])} methods. In case of successfully
	 * parsing this {@link Syntaxable} and / or the child {@link Syntaxable}
	 * instances, the evaluated command line arguments are returned: Depending
	 * on the {@link Syntaxable} subclasses representing the evaluated command
	 * line arguments, instances of {@link Flag} classes, {@link Option} classes
	 * or {@link Operand} classes may be found in the result. In case of parsing
	 * failure, an according exception is thrown. ATTENTION: This method does
	 * not test for superfluous command line arguments being passed; e.g.
	 * command line arguments not being evaluated by any of the
	 * {@link Syntaxable} instance being traversed. This method is commonly used
	 * by a root {@link Syntaxable}'s {@link ArgsParser#evalArgs(String[])}
	 * method, which delegates to the {@link #parseArgs(String[], String[])}
	 * method and after which it determines whether there are superfluous
	 * arguments to be taken care of (by throwing an according exception).
	 * Business logic therefore should invoke the root node's
	 * {@link #parseArgs(String[], String[])} method as ignoring superfluous
	 * command line arguments will cause unexpected behavior from the point of
	 * view of the invoker.
	 * 
	 * @param aArgs The command line arguments to be parsed.
	 * @param aOptions The list of options (short and as well as long) which are
	 *        reserved and cannot be used as value.
	 * 
	 * @return The list of evaluated command line arguments being instances of
	 *         the {@link Operand} interfaces or its sub-types.
	 * 
	 * @throws ArgsSyntaxException thrown in case of a command line arguments
	 *         mismatch regarding provided and expected args.
	 */
	List<? extends Operand<?>> parseArgs( String[] aArgs, String[] aOptions ) throws ArgsSyntaxException;

	/**
	 * Searches for a value being set for the given for given alias in the
	 * syntax tree parsing the command line arguments; if a node in the syntax
	 * tree has a value for the given alias, then this value is returned.
	 * 
	 * @param <V> The type of the value being expected.
	 * @param aAlias The alias for which to seek for a value being set.
	 * 
	 * @return The according value or null if none has been set for the given
	 *         alias.
	 */
	<V> V toValue( String aAlias );

	/**
	 * The {@link #toState()} method might be used by the
	 * {@link Object#toString()} method and prints out the state of the
	 * syntaxable and (if any) its children. Even in case parsing command line
	 * arguments failed, the state till failure can be inspected with this
	 * method.
	 * 
	 * @return The state of this {@link Syntaxable} instance and (if any) its
	 *         children.
	 */
	String toState();
}
