// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

import static org.junit.jupiter.api.Assertions.*;

import java.util.Arrays;

import org.junit.jupiter.api.Test;
import org.refcodes.textual.VerboseTextBuilder;

public class ClaimTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test" );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testClaims1() throws DependencyException {
		Reactor theReactor = new Reactor();
		Dependency<?> theA = theReactor.addDependency( ComponentA.class ).withAlias( "A" ).withAddClaim( ComponentA1.class, "A1" );
		theReactor.addDependency( ComponentA1.class ).withAlias( "A1" );
		theReactor.addDependency( ComponentA2.class ).withAlias( "A2" );
		Claim[] theClaims = theA.getClaims();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "################################################################################" );
			for ( Claim eClaim : theClaims ) {
				System.out.println( eClaim );
			}
		}
		assertEquals( 1, theClaims.length );
		assertEquals( "A1", theClaims[0].getAlias() );
		assertEquals( ComponentA1.class, theClaims[0].getType() );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		theA = theCtx.getDependencyByAlias( "A" );
		assertEquals( 1, theA.getClaims().length );
		assertEquals( "A1", theA.getClaims()[0].getAlias() );
		assertEquals( ComponentA1.class, theA.getClaims()[0].getType() );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testClaims2() throws DependencyException {
		Reactor theReactor = new Reactor();
		Dependency<?> theA = theReactor.addDependency( ComponentA.class ).withAlias( "A" ).withAddClaim( ComponentA1.class, "A1" ).withAddClaim( ComponentA2.class, "A2" );
		theReactor.addDependency( ComponentA1.class ).withAlias( "A1" );
		theReactor.addDependency( ComponentA2.class ).withAlias( "A2" );
		Claim[] theClaims = theA.getClaims();
		Arrays.sort( theClaims );
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "################################################################################" );
			for ( Claim eClaim : theClaims ) {
				System.out.println( eClaim );
			}
		}
		assertEquals( 2, theClaims.length );
		assertEquals( "A1", theClaims[0].getAlias() );
		assertEquals( ComponentA1.class, theClaims[0].getType() );
		assertEquals( "A2", theClaims[1].getAlias() );
		assertEquals( ComponentA2.class, theClaims[1].getType() );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		theA = theCtx.getDependencyByAlias( "A" );
		assertEquals( 2, theA.getClaims().length );
		theClaims = theA.getClaims();
		Arrays.sort( theClaims );
		assertEquals( "A1", theClaims[0].getAlias() );
		assertEquals( ComponentA1.class, theClaims[0].getType() );
		assertEquals( "A2", theClaims[1].getAlias() );
		assertEquals( ComponentA2.class, theClaims[1].getType() );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testMatchingClaims1() throws DependencyException {
		Reactor theReactor = new Reactor();
		Dependency<?> theQ = theReactor.addDependency( ComponentQ.class ).withAlias( "Q" ).withAddClaim( ComponentQ1.class, "Q1" );
		theReactor.addDependency( new ComponentQ1( "Q1, take this one" ) ).withAlias( "Q1" );
		theReactor.addDependency( new ComponentQ1( "Another Q1, don't pick it" ) );
		theReactor.addDependency( new ComponentQ2( "Q2" ) );
		Claim[] theClaims = theQ.getClaims();
		System.out.println( "################################################################################" );
		if ( IS_LOG_TEST_ENABLED ) {
			for ( Claim eClaim : theClaims ) {
				System.out.println( eClaim );
			}
		}
		assertEquals( 1, theClaims.length );
		assertEquals( "Q1", theClaims[0].getAlias() );
		assertEquals( ComponentQ1.class, theClaims[0].getType() );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		theQ = theCtx.getDependencyByAlias( "Q" );
		assertEquals( 1, theQ.getClaims().length );
		assertEquals( "Q1", theQ.getClaims()[0].getAlias() );
		assertEquals( ComponentQ1.class, theQ.getClaims()[0].getType() );
		ComponentQ theComponentQ = theCtx.getSingletonByAlias( "Q", ComponentQ.class );
		assertEquals( "Q1, take this one", theComponentQ.getQ1().getAlias() );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testUnsatisfiedClaimException1() throws DependencyException {
		Reactor theReactor = new Reactor();
		Dependency<?> theQ = theReactor.addDependency( ComponentQ.class ).withAlias( "Q" ).withAddClaim( ComponentQ1.class, "Q1" );
		theReactor.addDependency( new ComponentQ1( "Alias=Q1" ) );
		theReactor.addDependency( new ComponentQ2( "Alias=Q2" ) );
		Claim[] theClaims = theQ.getClaims();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "################################################################################" );
			for ( Claim eClaim : theClaims ) {
				System.out.println( eClaim );
			}
		}
		assertEquals( 1, theClaims.length );
		assertEquals( "Q1", theClaims[0].getAlias() );
		assertEquals( ComponentQ1.class, theClaims[0].getType() );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + UnsatisfiedClaimException.class.getName() + ">!" );
		}
		catch ( UnsatisfiedClaimException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private void log( Context aCtx ) {
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "################################################################################" );
			System.out.println( "PROFILES = " + VerboseTextBuilder.asString( aCtx.getProfiles() ) );
			System.out.println( "################################################################################" );
			for ( Dependency<?> eDependency : aCtx.getDependencies() ) {
				System.out.println( "DEPENDENCY: " + eDependency );
			}
			System.out.println( "--------------------------------------------------------------------------------" );
			boolean hasDependency = false;
			for ( Dependency<?> eDependency : aCtx.getDependencies() ) {
				for ( Object eObj : eDependency.getInstances() ) {
					hasDependency = true;
					System.out.println( eDependency.getInstanceMetrics() + ": " + eObj );
				}
			}
			if ( !hasDependency ) {
				System.out.println( "(no dependencies)" );
			}
			System.out.println( "--------------------------------------------------------------------------------" );
			System.out.println( aCtx.toSchema() );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
