// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

import static org.junit.jupiter.api.Assertions.*;

import org.junit.jupiter.api.Test;
import org.refcodes.textual.VerboseTextBuilder;

public class ContextTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test" );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testAmbigousDependencyException1() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAlias( "A1.1" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAlias( "A1.2" );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + AmbigousDependencyException.class.getName() + ">!" );
		}
		catch ( AmbigousDependencyException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousDependencyException2() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY ).withAddTag( "test" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAlias( "A1.1" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAlias( "A1.2" );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + AmbigousDependencyException.class.getName() + ">!" );
		}
		catch ( AmbigousDependencyException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousDependencyException3() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY ).withAddTag( "test" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddTag( "production" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + AmbigousDependencyException.class.getName() + ">!" );
		}
		catch ( AmbigousDependencyException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousDependencyException4() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY ).withAddTag( "test" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddTag( "test" ).withAlias( "A1.1" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddTag( "test" ).withAlias( "A1.2" );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + AmbigousDependencyException.class.getName() + ">!" );
		}
		catch ( AmbigousDependencyException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousDependencyOnDemand() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 0, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousSubClassDependency1() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentExtendsA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddProfile( "EXTENSIONS" );
		Context theCtx = theReactor.createContext( "EXTENSIONS" );
		log( theCtx );
		assertEquals( 3, theCtx.getInstances().length );
		ComponentA2 theA2 = theCtx.getFirstByType( ComponentA2.class );
		assertTrue( theA2 instanceof ComponentExtendsA2 );
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( ((ComponentExtendsA2) theA2).whoAmI() );
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousSubClassDependency2() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddProfile( "EXTENSIONS" );
		theReactor.addDependency( ComponentExtendsA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext( "EXTENSIONS" );
		log( theCtx );
		assertEquals( 3, theCtx.getInstances().length );
		ComponentA2 theA2 = theCtx.getFirstByType( ComponentA2.class );
		assertFalse( theA2 instanceof ComponentExtendsA2 );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousDependencyByAlias() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAlias( "a1" );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA2.class ).length );
		assertEquals( 3, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousDependencyWithProfile() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY ).withAddProfile( "test" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddProfile( "prod" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddProfile( "test" );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddProfile( "test" );
		Context theCtx = theReactor.createContext( "test" );
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA2.class ).length );
		assertEquals( 3, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousDependencyWithTag1() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY ).withAddTag( "test" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddTag( "test" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddTag( "test" );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA2.class ).length );
		assertEquals( 3, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testAmbigousDependencyWithTag2() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY ).withAddTag( "test" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND ).withAddTag( "test" );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA2.class ).length );
		assertEquals( 3, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testCircularDependency1() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( From1To2To3.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( From2To3To1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( From3To1To2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( From1To2To3.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( From2To3To1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( From3To1To2.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 4, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testCircularDependencyException1() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( From1To2.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( From2To1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + CircularDependencyException.class.getName() + ">!" );
		}
		catch ( CircularDependencyException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testCircularDependencyException2() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( From1To2To3.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( From2To3To1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( From3To1To2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + CircularDependencyException.class.getName() + ">!" );
		}
		catch ( CircularDependencyException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testContext1() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentX.class );
		theReactor.addDependency( ComponentA.class );
		theReactor.addDependency( ComponentB.class );
		theReactor.addDependency( ComponentA1.class );
		theReactor.addDependency( ComponentA2.class );
		theReactor.addDependency( ComponentB1.class );
		theReactor.addDependency( ComponentB2.class );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( ComponentX.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentB.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA2.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentB1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentB2.class ).length );
		assertEquals( 7, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testContext2() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentX.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.SINGLETON_ON_DEMAND );
		theReactor.addDependency( ComponentB.class ).withInstanceMetrics( InstanceMode.SINGLETON_ON_DEMAND );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.SINGLETON_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.SINGLETON_ON_DEMAND );
		theReactor.addDependency( ComponentB1.class ).withInstanceMetrics( InstanceMode.SINGLETON_ON_DEMAND );
		theReactor.addDependency( ComponentB2.class ).withInstanceMetrics( InstanceMode.SINGLETON_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( ComponentX.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentB.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA2.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentB1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentB2.class ).length );
		assertEquals( 7, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testContext3() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentX.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentB.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentB1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentB2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( ComponentX.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentB.class ).length );
		assertEquals( 2, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 2, theCtx.getInstancesByType( ComponentA2.class ).length );
		assertEquals( 2, theCtx.getInstancesByType( ComponentB1.class ).length );
		assertEquals( 2, theCtx.getInstancesByType( ComponentB2.class ).length );
		assertEquals( 11, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testContext4() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentX.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentY.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentB.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentB1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentB2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( ComponentX.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentY.class ).length );
		assertEquals( 2, theCtx.getInstancesByType( ComponentA.class ).length );
		assertEquals( 2, theCtx.getInstancesByType( ComponentB.class ).length );
		assertEquals( 4, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 4, theCtx.getInstancesByType( ComponentA2.class ).length );
		assertEquals( 4, theCtx.getInstancesByType( ComponentB1.class ).length );
		assertEquals( 4, theCtx.getInstancesByType( ComponentB2.class ).length );
		assertEquals( 22, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testContext5() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentX.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentY.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentB.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.SINGLETON_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentB1.class ).withInstanceMetrics( InstanceMode.SINGLETON_ON_DEMAND );
		theReactor.addDependency( ComponentB2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		assertEquals( 1, theCtx.getInstancesByType( ComponentX.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentY.class ).length );
		assertEquals( 2, theCtx.getInstancesByType( ComponentA.class ).length );
		assertEquals( 2, theCtx.getInstancesByType( ComponentB.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentA1.class ).length );
		assertEquals( 4, theCtx.getInstancesByType( ComponentA2.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentB1.class ).length );
		assertEquals( 4, theCtx.getInstancesByType( ComponentB2.class ).length );
		assertEquals( 16, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testDuplicateDependencyException1() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA1.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentA2.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + DuplicateDependencyException.class.getName() + ">!" );
		}
		catch ( DuplicateDependencyException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testUnsatisfiedDependenciesException1() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentX.class );
		theReactor.addDependency( ComponentA.class );
		theReactor.addDependency( ComponentB.class );
		theReactor.addDependency( ComponentA1.class );
		theReactor.addDependency( ComponentB1.class );
		theReactor.addDependency( ComponentB2.class );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + UnsatisfiedDependencyException.class.getName() + ">!" );
		}
		catch ( UnsatisfiedDependencyException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testUnsatisfiedDependenciesException2() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + UnsatisfiedDependencyException.class.getName() + ">!" );
		}
		catch ( UnsatisfiedDependencyException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testUnsatisfiedDependenciesOnDemand() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentA.class ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private void log( Context aCtx ) {
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "################################################################################" );
			System.out.println( "PROFILES = " + VerboseTextBuilder.asString( aCtx.getProfiles() ) );
			System.out.println( "################################################################################" );
			for ( Dependency<?> eDependency : aCtx.getDependencies() ) {
				System.out.println( "DEPENDENCY: " + eDependency );
			}
			System.out.println( "--------------------------------------------------------------------------------" );
			boolean hasDependency = false;
			for ( Dependency<?> eDependency : aCtx.getDependencies() ) {
				for ( Object eObj : eDependency.getInstances() ) {
					hasDependency = true;
					System.out.println( eDependency.getInstanceMetrics() + ": " + eObj );
				}
			}
			if ( !hasDependency ) {
				System.out.println( "(no dependencies)" );
			}
			System.out.println( "--------------------------------------------------------------------------------" );
			System.out.println( aCtx.toSchema() );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
