// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

import static org.junit.jupiter.api.Assertions.*;

import org.junit.jupiter.api.Test;
import org.refcodes.textual.VerboseTextBuilder;

public class FactoryTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test" );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testFactoryClaim() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentQ.class ).withAlias( "Q" ).withAddClaim( ComponentQ1.class, "Q1" ).withAddClaim( ComponentQ2.class, "Q2" );
		theReactor.addDependency( ComponentQ1.class ).withAlias( "Q1" ).withFactory( ComponentQ2.class, ( d ) -> { return new ComponentQ1( "Hello " + d.getAlias() ); } );
		theReactor.addDependency( new ComponentQ2( "ComponentQ2" ) ).withAlias( "Q2" );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
		ComponentQ1 theQ1 = theCtx.getFirstByType( ComponentQ1.class );
		assertEquals( "Hello ComponentQ2", theQ1.getAlias() );
	}

	@Test
	public void testUnsatisfiedFactoryException() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentQ.class ).withAlias( "Q" ).withAddClaim( ComponentQ1.class, "Q1" ).withAddClaim( ComponentQ2.class, "Q2" );
		theReactor.addDependency( ComponentQ1.class ).withAlias( "Q1" ).withFactory( ComponentQ2.class, ( d ) -> { return new ComponentQ1( "Hello " + d.getAlias() ); }, "XYZ" );
		theReactor.addDependency( new ComponentQ2( "ComponentQ2" ) ).withAlias( "Q2" );
		try {
			Context theCtx = theReactor.createContext();
			log( theCtx );
			fail( "Expecting an exception of type <" + UnsatisfiedFactoryException.class.getName() + ">!" );
		}
		catch ( UnsatisfiedFactoryException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "################################################################################" );
				System.out.println( "EXPECTED: " + e.getMessage() );
				System.out.println( "################################################################################" );
			}
		}
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	@Test
	public void testFactoryFromDependency1() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentQ.class, QFactory.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( new ComponentQ1( "Q1" ) );
		theReactor.addDependency( new ComponentQ2( "Q2" ) );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
		ComponentQ theQ = theCtx.getFirstByType( ComponentQ.class );
		assertEquals( "Q1+Q2", theQ.getAlias() );
		assertNull( theQ.getQ1() );
		assertNull( theQ.getQ2() );

	}

	@Test
	public void testFactoryFromDependency2() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentQ.class, QFactory.class, ( f ) -> f.fabricateQ() ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( new ComponentQ1( "Q1" ) );
		theReactor.addDependency( new ComponentQ2( "Q2" ) );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
		ComponentQ theQ = theCtx.getFirstByType( ComponentQ.class );
		assertEquals( "Q1+Q2", theQ.getAlias() );
		assertNull( theQ.getQ1() );
		assertNull( theQ.getQ2() );

	}

	@Test
	public void testFactoryFromDependencyFacty() throws DependencyException {
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentQ.class, QFactory.class ).withInstanceMetrics( InstanceMode.INSTANCE_IS_MANDATORY );
		theReactor.addDependency( ComponentQ1.class, () -> new ComponentQ1( "Q1" ) ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		theReactor.addDependency( ComponentQ2.class, () -> new ComponentQ2( "Q2" ) ).withInstanceMetrics( InstanceMode.INSTANCE_ON_DEMAND );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
		ComponentQ theQ = theCtx.getFirstByType( ComponentQ.class );
		assertEquals( "Q1+Q2", theQ.getAlias() );
		assertNull( theQ.getQ1() );
		assertNull( theQ.getQ2() );

	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private void log( Context aCtx ) {
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "################################################################################" );
			System.out.println( "PROFILES = " + VerboseTextBuilder.asString( aCtx.getProfiles() ) );
			System.out.println( "################################################################################" );
			for ( Dependency<?> eDependency : aCtx.getDependencies() ) {
				System.out.println( "DEPENDENCY: " + eDependency );
			}
			System.out.println( "--------------------------------------------------------------------------------" );
			boolean hasDependency = false;
			for ( Dependency<?> eDependency : aCtx.getDependencies() ) {
				for ( Object eObj : eDependency.getInstances() ) {
					hasDependency = true;
					System.out.println( eDependency.getInstanceMetrics() + ": " + eObj );
				}
			}
			if ( !hasDependency ) {
				System.out.println( "(no dependencies)" );
			}
			System.out.println( "--------------------------------------------------------------------------------" );
			System.out.println( aCtx.toSchema() );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
