// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.decoupling;

import static org.junit.jupiter.api.Assertions.*;

import org.junit.jupiter.api.Test;
import org.refcodes.textual.VerboseTextBuilder;

public class InterceptorTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test" );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testInterceptors() throws DependencyException {
		int[] count = { 0, 0 };
		String theQ2Alias = "Interceptor's alias!";
		Reactor theReactor = new Reactor();
		theReactor.addDependency( ComponentQ.class );
		theReactor.addDependency( ComponentQ1.class );
		theReactor.addDependency( ComponentQ2.class );
		theReactor.addInterceptor( ( t, d ) -> { count[0]++; return t; } );
		theReactor.addInterceptor( ( t, d ) -> {
			count[1]++;
			if ( t instanceof ComponentQ2 q2 ) {
				q2.setAlias( theQ2Alias );
			}
			return t;
		} );
		Context theCtx = theReactor.createContext();
		log( theCtx );
		ComponentQ2 theQ2 = theCtx.getFirstByType( ComponentQ2.class );
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Interceptor 1 invoked <" + count[0] + "> times ..." );
			System.out.println( "Interceptor 2 invoked <" + count[1] + "> times ..." );
			System.out.println( "The Q2's alias = \"" + theQ2.getAlias() + "\" ..." );
		}
		assertEquals( theQ2Alias, theQ2.getAlias() );
		assertEquals( 3, count[0] );
		assertEquals( 3, count[1] );
		assertEquals( 1, theCtx.getInstancesByType( ComponentQ.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentQ1.class ).length );
		assertEquals( 1, theCtx.getInstancesByType( ComponentQ2.class ).length );
		assertEquals( 3, theCtx.getInstances().length );
		if ( IS_LOG_TEST_ENABLED ) System.out.println();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private void log( Context aCtx ) {
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "################################################################################" );
			System.out.println( "PROFILES = " + VerboseTextBuilder.asString( aCtx.getProfiles() ) );
			System.out.println( "################################################################################" );
			for ( Dependency<?> eDependency : aCtx.getDependencies() ) {
				System.out.println( "DEPENDENCY: " + eDependency );
			}
			System.out.println( "--------------------------------------------------------------------------------" );
			boolean hasDependency = false;
			for ( Dependency<?> eDependency : aCtx.getDependencies() ) {
				for ( Object eObj : eDependency.getInstances() ) {
					hasDependency = true;
					System.out.println( eDependency.getInstanceMetrics() + ": " + eObj );
				}
			}
			if ( !hasDependency ) {
				System.out.println( "(no dependencies)" );
			}
			System.out.println( "--------------------------------------------------------------------------------" );
			System.out.println( aCtx.toSchema() );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
