package org.sqlproc.engine;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sqlproc.engine.impl.SqlEmptyMonitor;
import org.sqlproc.engine.impl.SqlMappingRule;
import org.sqlproc.engine.impl.SqlMetaStatement;
import org.sqlproc.engine.plugin.SimpleSqlPluginFactory;
import org.sqlproc.engine.plugin.SqlPluginFactory;
import org.sqlproc.engine.type.SqlTypeFactory;

/**
 * Common ancestor for {@link SqlQueryEngine} and {@link SqlCrudEngine}.
 * 
 * <p>
 * For more info please see the <a href="https://github.com/hudec/sql-processor/wiki">Tutorials</a>.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public abstract class SqlEngine {

    /**
     * The internal slf4j logger.
     */
    protected final Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * Name of the META SQL query or statement, which uniquely identifies this instance.
     */
    protected String name;

    /**
     * The pre-compiled META SQL query or statement. The META SQL is an ANSI SQL extension using the ANTLR defined
     * grammar.
     */
    protected SqlMetaStatement statement;

    /**
     * The pre-compiled mapping rule, which is an SQL execution result to Java output classes mapping prescription.
     */
    protected SqlMappingRule mapping;

    /**
     * Configuration of the SQL Processor using map of features. Optional features can alter the SQL Processor runtime
     * behavior.
     */
    protected Map<String, Object> features = new HashMap<String, Object>();

    /**
     * Monitor for the runtime statistics gathering.
     */
    protected SqlMonitor monitor;

    /**
     * The factory for the META types construction. The META type defines the mapping between a Java class type and a
     * JDBC datatype. For the default META type factory, please see {@link org.sqlproc.engine.jdbc.type.JdbcTypeFactory}
     * .
     */
    protected SqlTypeFactory typeFactory;

    /**
     * The factory for the SQL Processor plugins. This is the basic facility to alter the SQL Processor processing.
     */
    protected SqlPluginFactory pluginFactory;

    /**
     * Creates a new instance of the SqlEngine from one META SQL statement and one SQL Mapping rule instance. Both
     * parameters are already pre-compiled instances using the ANTLR parsers. This is the recommended usage for the
     * runtime performance optimization. This constructor is devoted to be used from the {@link SqlProcessorLoader},
     * which is able to read all statements definitions from an external meta statements file and create the named
     * SqlEngine instances. Also an external SQL Monitor for the runtime statistics gathering can be engaged.
     * 
     * @param name
     *            the name of this SQL Engine instance
     * @param statement
     *            the pre-compiled META SQL statement
     * @param mapping
     *            the pre-compiled SQL mapping rule
     * @param monitor
     *            the SQL Monitor for the runtime statistics gathering
     * @param features
     *            the optional SQL Processor features
     * @param typeFactory
     *            the factory for the META types construction
     * @param pluginFactory
     *            the factory for the SQL Processor plugins
     */
    public SqlEngine(String name, SqlMetaStatement statement, SqlMappingRule mapping, SqlMonitor monitor,
            Map<String, Object> features, SqlTypeFactory typeFactory, SqlPluginFactory pluginFactory) {
        this.name = name;
        this.statement = statement;
        this.mapping = mapping;
        if (features != null) {
            this.features.putAll(features);
        }
        this.monitor = (monitor != null) ? monitor : new SqlEmptyMonitor();
        this.typeFactory = typeFactory;
        this.pluginFactory = (pluginFactory != null) ? pluginFactory : SimpleSqlPluginFactory.getInstance();
    }

    /**
     * Sets the optional feature in the runtime.
     * 
     * @param name
     *            the name of the optional feature
     * @param value
     *            the value of the optional feature
     */
    public void setFeature(String name, Object value) {
        features.put(name, value);
        if (SqlFeature.SURROUND_QUERY_LIKE_FULL.equals(name) || SqlFeature.SURROUND_QUERY_LIKE.equals(name)) {
            unsetFeature(SqlFeature.SURROUND_QUERY_LIKE_PARTIAL);
        } else if (SqlFeature.SURROUND_QUERY_LIKE_PARTIAL.equals(name)) {
            unsetFeature(SqlFeature.SURROUND_QUERY_LIKE_FULL);
            unsetFeature(SqlFeature.SURROUND_QUERY_LIKE);
        } else if (SqlFeature.EMPTY_FOR_NULL.equals(name)) {
            unsetFeature(SqlFeature.EMPTY_USE_METHOD_IS_NULL);
        } else if (SqlFeature.EMPTY_USE_METHOD_IS_NULL.equals(name)) {
            unsetFeature(SqlFeature.EMPTY_FOR_NULL);
        }
    }

    /**
     * Clears the optional feature in the runtime.
     * 
     * @param name
     *            the name of the optional feature
     */
    public void unsetFeature(String name) {
        features.remove(name);
    }

    /**
     * The helper to prevent the NPE
     * 
     * @param sqlControl
     *            the compound parameters controlling the META SQL execution
     * @return the object used for the SQL statement static input values
     */
    Object getStaticInputValues(SqlControl sqlControl) {
        if (sqlControl == null)
            return null;
        else
            return sqlControl.getStaticInputValues();
    }

    /**
     * The helper to prevent the NPE
     * 
     * @param sqlControl
     *            the compound parameters controlling the META SQL execution
     * @return the max SQL execution time
     */
    int getMaxTimeout(SqlControl sqlControl) {
        if (sqlControl == null)
            return 0;
        else
            return sqlControl.getMaxTimeout();
    }

    /**
     * The helper to prevent the NPE
     * 
     * @param sqlControl
     *            the compound parameters controlling the META SQL execution
     * @return the first SQL execution output row
     */
    int getFirstResult(SqlControl sqlControl) {
        if (sqlControl == null)
            return 0;
        else
            return sqlControl.getFirstResult();
    }

    /**
     * The helper to prevent the NPE
     * 
     * @param sqlControl
     *            the compound parameters controlling the META SQL execution
     * @return the max number of SQL execution output rows
     */
    int getMaxResults(SqlControl sqlControl) {
        if (sqlControl == null)
            return 0;
        else
            return sqlControl.getMaxResults();
    }

    /**
     * The helper to prevent the NPE
     * 
     * @param sqlControl
     *            the compound parameters controlling the META SQL execution
     * @return the ordering directive list
     */
    SqlOrder getOrder(SqlControl sqlControl) {
        if (sqlControl == null || sqlControl.getOrder() == null)
            return SqlQueryEngine.NO_ORDER;
        else
            return sqlControl.getOrder();
    }

    /**
     * The helper to prevent the NPE
     * 
     * @param sqlControl
     *            the compound parameters controlling the META SQL execution
     * @return more result classes used for the return values
     */
    Map<String, Class<?>> getMoreResultClasses(SqlControl sqlControl) {
        if (sqlControl == null)
            return null;
        else
            return sqlControl.getMoreResultClasses();
    }
}