/*
 * Decompiled with CFR 0.152.
 */
package com.devcycle.sdk.server.local.managers;

import com.devcycle.sdk.server.common.api.IDevCycleApi;
import com.devcycle.sdk.server.common.exception.DevCycleException;
import com.devcycle.sdk.server.common.logging.DevCycleLogger;
import com.devcycle.sdk.server.common.model.ErrorResponse;
import com.devcycle.sdk.server.common.model.HttpResponseCode;
import com.devcycle.sdk.server.common.model.ProjectConfig;
import com.devcycle.sdk.server.common.model.SSEMessage;
import com.devcycle.sdk.server.local.api.DevCycleLocalApiClient;
import com.devcycle.sdk.server.local.bucketing.LocalBucketing;
import com.devcycle.sdk.server.local.managers.DaemonThreadFactory;
import com.devcycle.sdk.server.local.managers.SSEManager;
import com.devcycle.sdk.server.local.model.DevCycleLocalOptions;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.launchdarkly.eventsource.FaultEvent;
import com.launchdarkly.eventsource.MessageEvent;
import com.launchdarkly.eventsource.StartedEvent;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import retrofit2.Call;
import retrofit2.Response;

public final class EnvironmentConfigManager {
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final int DEFAULT_POLL_INTERVAL_MS = 30000;
    private static final int MIN_INTERVALS_MS = 1000;
    private ScheduledExecutorService scheduler;
    private final IDevCycleApi configApiClient;
    private final LocalBucketing localBucketing;
    private SSEManager sseManager;
    private boolean isSSEConnected = false;
    private final DevCycleLocalOptions options;
    private ProjectConfig config;
    private String configETag = "";
    private String configLastModified = "";
    private final String sdkKey;
    private final int pollingIntervalMS;
    private static final int pollingIntervalSSEMS = 54000000;
    private boolean pollingEnabled = true;
    private final Runnable getConfigRunnable = new Runnable(){

        @Override
        public void run() {
            try {
                if (EnvironmentConfigManager.this.pollingEnabled) {
                    EnvironmentConfigManager.this.getConfig();
                }
            }
            catch (DevCycleException e) {
                DevCycleLogger.error("Failed to load config: " + e.getMessage());
            }
        }
    };

    public EnvironmentConfigManager(String sdkKey, LocalBucketing localBucketing, DevCycleLocalOptions options) {
        this.sdkKey = sdkKey;
        this.localBucketing = localBucketing;
        this.options = options;
        this.configApiClient = new DevCycleLocalApiClient(sdkKey, options).initialize();
        int configPollingIntervalMS = options.getConfigPollingIntervalMS();
        this.pollingIntervalMS = configPollingIntervalMS >= 1000 ? configPollingIntervalMS : 30000;
        this.scheduler = this.setupScheduler();
        this.scheduler.scheduleAtFixedRate(this.getConfigRunnable, 0L, this.pollingIntervalMS, TimeUnit.MILLISECONDS);
    }

    private ScheduledExecutorService setupScheduler() {
        return Executors.newScheduledThreadPool(1, new DaemonThreadFactory());
    }

    public boolean isConfigInitialized() {
        return this.config != null;
    }

    private ProjectConfig getConfig() throws DevCycleException {
        Call<ProjectConfig> config = this.configApiClient.getConfig(this.sdkKey, this.configETag, this.configLastModified);
        this.config = this.getResponseWithRetries(config, 1);
        if (this.options.isEnableBetaRealtimeUpdates()) {
            try {
                URI uri = new URI(this.config.getSse().getHostname() + this.config.getSse().getPath());
                if (this.sseManager == null) {
                    this.sseManager = new SSEManager(uri);
                }
                this.sseManager.restart(uri, this::handleSSEMessage, this::handleSSEError, this::handleSSEStarted);
            }
            catch (URISyntaxException e) {
                DevCycleLogger.warning("Failed to create SSEManager: " + e.getMessage());
            }
        }
        return this.config;
    }

    private Void handleSSEMessage(MessageEvent messageEvent) {
        String data;
        DevCycleLogger.debug("Received message: " + messageEvent.getData());
        if (!this.isSSEConnected) {
            this.handleSSEStarted(null);
        }
        if ((data = messageEvent.getData()) == null || data.isEmpty() || data.equals("keepalive")) {
            return null;
        }
        try {
            SSEMessage message = (SSEMessage)OBJECT_MAPPER.readValue(data, SSEMessage.class);
            if (message.getType() == null || message.getType().equals("refetchConfig") || message.getType().isEmpty()) {
                DevCycleLogger.debug("Received refetchConfig message, fetching new config");
                this.getConfigRunnable.run();
            }
        }
        catch (JsonProcessingException e) {
            DevCycleLogger.warning("Failed to parse SSE message: " + e.getMessage());
        }
        return null;
    }

    private Void handleSSEError(FaultEvent faultEvent) {
        DevCycleLogger.warning("Received error: " + faultEvent.getCause());
        return null;
    }

    private Void handleSSEStarted(StartedEvent startedEvent) {
        this.isSSEConnected = true;
        DevCycleLogger.debug("SSE Connected - setting polling interval to 54000000");
        this.scheduler.shutdown();
        this.scheduler = this.setupScheduler();
        this.scheduler.scheduleAtFixedRate(this.getConfigRunnable, 0L, 54000000L, TimeUnit.MILLISECONDS);
        return null;
    }

    private ProjectConfig getResponseWithRetries(Call<ProjectConfig> call, int maxRetries) throws DevCycleException {
        int attempt = 0;
        while (true) {
            try {
                return this.getConfigResponse(call);
            }
            catch (DevCycleException e) {
                if (!e.isRetryable() || ++attempt > maxRetries) {
                    throw e;
                }
                try {
                    long waitIntervalMS = (long)(10.0 * Math.pow(2.0, attempt));
                    Thread.sleep(waitIntervalMS);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                call = call.clone();
                if (attempt <= maxRetries && this.pollingEnabled) continue;
                ErrorResponse errorResponse = ErrorResponse.builder().build();
                errorResponse.setMessage("Out of retry attempts");
                throw new DevCycleException(HttpResponseCode.SERVER_ERROR, errorResponse);
            }
            break;
        }
    }

    private ProjectConfig getConfigResponse(Call<ProjectConfig> call) throws DevCycleException {
        Response response;
        ErrorResponse errorResponse = ErrorResponse.builder().build();
        try {
            response = call.execute();
        }
        catch (JsonParseException badJsonExc) {
            errorResponse.setMessage(badJsonExc.getMessage());
            throw new DevCycleException(HttpResponseCode.NO_CONTENT, errorResponse);
        }
        catch (IOException e) {
            errorResponse.setMessage(e.getMessage());
            throw new DevCycleException(HttpResponseCode.byCode(500), errorResponse);
        }
        HttpResponseCode httpResponseCode = HttpResponseCode.byCode(response.code());
        errorResponse.setMessage("Unknown error");
        if (response.isSuccessful()) {
            ZonedDateTime configLastModified;
            ZonedDateTime parsedLastModified;
            String currentETag = response.headers().get("ETag");
            String headerLastModified = response.headers().get("Last-Modified");
            if (!this.configLastModified.isEmpty() && headerLastModified != null && !headerLastModified.isEmpty() && (parsedLastModified = ZonedDateTime.parse(headerLastModified, DateTimeFormatter.RFC_1123_DATE_TIME)).isBefore(configLastModified = ZonedDateTime.parse(this.configLastModified, DateTimeFormatter.RFC_1123_DATE_TIME))) {
                DevCycleLogger.warning("Received a config with last-modified header before the current stored timestamp. Not saving config.");
                return this.config;
            }
            ProjectConfig config = (ProjectConfig)response.body();
            try {
                ObjectMapper mapper = new ObjectMapper();
                this.localBucketing.storeConfig(this.sdkKey, mapper.writeValueAsString((Object)config));
            }
            catch (JsonProcessingException e) {
                if (this.config != null) {
                    DevCycleLogger.error("Unable to parse config with etag: " + currentETag + ". Using cache, etag " + this.configETag + " last-modified: " + this.configLastModified);
                    return this.config;
                }
                errorResponse.setMessage(e.getMessage());
                throw new DevCycleException(HttpResponseCode.SERVER_ERROR, errorResponse);
            }
            this.configETag = currentETag;
            this.configLastModified = headerLastModified;
            return (ProjectConfig)response.body();
        }
        if (httpResponseCode == HttpResponseCode.NOT_MODIFIED) {
            DevCycleLogger.debug("Config not modified, using cache, etag: " + this.configETag + " last-modified: " + this.configLastModified);
            return this.config;
        }
        if (response.errorBody() != null) {
            try {
                errorResponse = (ErrorResponse)OBJECT_MAPPER.readValue(response.errorBody().string(), ErrorResponse.class);
            }
            catch (JsonProcessingException e) {
                errorResponse.setMessage("Unable to parse error response: " + e.getMessage());
                throw new DevCycleException(httpResponseCode, errorResponse);
            }
            catch (IOException e) {
                errorResponse.setMessage(e.getMessage());
                throw new DevCycleException(httpResponseCode, errorResponse);
            }
            throw new DevCycleException(httpResponseCode, errorResponse);
        }
        if (httpResponseCode == HttpResponseCode.UNAUTHORIZED || httpResponseCode == HttpResponseCode.FORBIDDEN) {
            errorResponse.setMessage("API Key is unauthorized");
            this.stopPolling();
        } else if (!response.message().equals("")) {
            try {
                errorResponse = (ErrorResponse)OBJECT_MAPPER.readValue(response.message(), ErrorResponse.class);
            }
            catch (JsonProcessingException e) {
                errorResponse.setMessage(e.getMessage());
                throw new DevCycleException(httpResponseCode, errorResponse);
            }
        }
        throw new DevCycleException(httpResponseCode, errorResponse);
    }

    private void stopPolling() {
        this.pollingEnabled = false;
        this.scheduler.shutdown();
    }

    public void cleanup() {
        if (this.sseManager != null) {
            this.sseManager.close();
        }
        this.stopPolling();
    }
}

