/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.expression.LambdaParameter;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.TypeName;
import java.util.Objects;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeMirror;

public class ParameterCode extends AbstractCodeElement
    implements
    LambdaParameter {

  private final TypeName typeName;
  private final String name;

  ParameterCode(TypeName typeName, String name) {
    this.typeName = typeName;
    this.name = name;
  }

  public static ParameterCode of(VariableElement element) {
    checkNotNull(element, "element == null");
    TypeMirror type = element.asType();
    return new ParameterCode(
        TypeName.ofUnchecked(type),
        element.getSimpleName().toString());
  }

  public static ParameterCode of(Class<?> type, String name) {
    ClassName typeName = ClassName.of(type);
    checkNotNull(name, "name == null");
    return new ParameterCode(typeName, name);
  }

  public static ParameterCode of(TypeName typeName, String name) {
    checkNotNull(typeName, "typeName == null");
    checkNotNull(name, "name == null");
    return new ParameterCode(typeName, name);
  }

  static ParameterCode ofUnchecked(TypeName typeName, String name) {
    return new ParameterCode(typeName, name);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter writer) {
    return writer.writeTypeNameAsWord(typeName).writeWord(name);
  }

  @Override
  public final boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (!(obj instanceof ParameterCode)) {
      return false;
    }
    ParameterCode that = (ParameterCode) obj;
    return Objects.equals(typeName, that.typeName)
        && Objects.equals(name, that.name);
  }

  @Override
  public final int hashCode() {
    return Objects.hash(typeName, name);
  }

  public final String name() {
    return name;
  }

  public final TypeName typeName() {
    return typeName;
  }

}