/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.statement;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.expression.ExpressionCode;
import br.com.objectos.code.java.expression.Identifier;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.TypeName;

public class EnhancedForStatement extends AbstractStatement implements ForStatement {

  private final TypeName typeName;
  private final Identifier id;
  private final ExpressionCode expression;
  private final Statement statement;

  private EnhancedForStatement(TypeName typeName,
                               Identifier id,
                               ExpressionCode expression,
                               Statement statement) {
    this.typeName = typeName;
    this.id = id;
    this.expression = expression;
    this.statement = statement;
  }

  static Builder builder(Class<?> type, Identifier id, ExpressionCode expression) {
    checkNotNull(type, "type == null");
    checkNotNull(id, "id == null");
    checkNotNull(expression, "expression == null");
    ClassName className = ClassName.of(type);
    return new Builder(className, id, expression);
  }

  static Builder builder(TypeName typeName, Identifier id, ExpressionCode expression) {
    checkNotNull(typeName, "typeName == null");
    checkNotNull(id, "id == null");
    checkNotNull(expression, "expression == null");
    return new Builder(typeName, id, expression);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w
        .writeWord("for")
        .writeWord('(').spaceOff()
        .writeTypeNameAsWord(typeName)
        .writeWord(id.name())
        .writeWord(':')
        .writeCodeElement(expression)
        .spaceOff().writeWord(')')
        .writeCodeElement(statement);
  }

  public static class Builder {

    private final TypeName typeName;
    private final Identifier id;
    private final ExpressionCode expression;

    private Builder(TypeName typeName, Identifier id, ExpressionCode expression) {
      this.typeName = typeName;
      this.id = id;
      this.expression = expression;
    }

    public final EnhancedForStatement block(BlockElement... elements) {
      Block block = Block.of(elements);
      return new EnhancedForStatement(typeName, id, expression, block);
    }

  }

}
