/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.type;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.expression.CastExpression;
import br.com.objectos.code.java.expression.Expressions;
import br.com.objectos.code.java.expression.MethodReference;
import br.com.objectos.code.java.expression.TypeWitness;
import br.com.objectos.code.java.expression.UnaryExpressionNotPlusMinus;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.io.JavaFileImportSet;
import java.util.Objects;

public abstract class ParameterizedTypeName implements ClassNameOrParameterizedTypeName {

  ParameterizedTypeName() {}

  public static ParameterizedTypeName of(ClassName className, TypeName typeArgument) {
    checkNotNull(className, "className == null");
    checkNotNull(typeArgument, "typeArgument == null");
    return new Head(className, typeArgument);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w.writeTypeNameAsWord(this);
  }

  @Override
  public final String acceptJavaFileImportSet(JavaFileImportSet set) {
    return toStringBuilder(set)
        .append('>')
        .toString();
  }

  @Override
  public final <R, P> R acceptTypeNameVisitor(TypeNameVisitor<R, P> visitor, P p) {
    return visitor.visitParameterizedTypeName(this, p);
  }

  @Override
  public final CastExpression cast(UnaryExpressionNotPlusMinus expression) {
    return Expressions.cast(this, expression);
  }

  @Override
  public final boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (!(obj instanceof ParameterizedTypeName)) {
      return false;
    }
    ParameterizedTypeName that = (ParameterizedTypeName) obj;
    return getClass().equals(that.getClass())
        && equalsImpl(that);
  }

  @Override
  public abstract int hashCode();

  @Override
  public final String toString() {
    return toStringBuilder()
        .append('>')
        .toString();
  }

  @Override
  public final MethodReference ref(String methodName) {
    return Expressions.ref(this, methodName);
  }

  @Override
  public final MethodReference ref(TypeWitness witness, String methodName) {
    return Expressions.ref(this, witness, methodName);
  }

  @Override
  public final ParameterizedTypeName withTypeArgument(TypeName typeName) {
    checkNotNull(typeName, "typeName == null");
    return new Node(this, typeName);
  }

  abstract boolean equalsImpl(ParameterizedTypeName typeName);

  abstract StringBuilder toStringBuilder();

  abstract StringBuilder toStringBuilder(JavaFileImportSet set);

  private static class Head extends ParameterizedTypeName {

    private final ClassName className;
    private final TypeName typeArgument;

    private Head(ClassName className, TypeName typeArgument) {
      this.className = className;
      this.typeArgument = typeArgument;
    }

    @Override
    public final TypeName arrayCreationTypeName() {
      return className;
    }

    @Override
    public final int hashCode() {
      return Objects.hash(className, typeArgument);
    }

    @Override
    final boolean equalsImpl(ParameterizedTypeName typeName) {
      Head that = (Head) typeName;
      return className.equals(that.className)
          && typeArgument.equals(that.typeArgument);
    }

    @Override
    final StringBuilder toStringBuilder() {
      return new StringBuilder()
          .append(className)
          .append('<')
          .append(typeArgument);
    }

    @Override
    final StringBuilder toStringBuilder(JavaFileImportSet set) {
      return new StringBuilder()
          .append(className.acceptJavaFileImportSet(set))
          .append('<')
          .append(typeArgument.acceptJavaFileImportSet(set));
    }

  }

  private static class Node extends ParameterizedTypeName {

    private final ParameterizedTypeName previous;
    private final TypeName typeArgument;

    private Node(ParameterizedTypeName previous, TypeName typeArgument) {
      this.previous = previous;
      this.typeArgument = typeArgument;
    }

    @Override
    public final TypeName arrayCreationTypeName() {
      return previous.arrayCreationTypeName();
    }

    @Override
    public final int hashCode() {
      return Objects.hash(previous, typeArgument);
    }

    @Override
    final boolean equalsImpl(ParameterizedTypeName typeName) {
      Node that = (Node) typeName;
      return previous.equals(that.previous)
          && typeArgument.equals(that.typeArgument);
    }

    @Override
    final StringBuilder toStringBuilder() {
      return previous.toStringBuilder()
          .append(", ")
          .append(typeArgument);
    }

    @Override
    final StringBuilder toStringBuilder(JavaFileImportSet set) {
      return previous.toStringBuilder(set)
          .append(", ")
          .append(typeArgument.acceptJavaFileImportSet(set));
    }

  }

}