/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.declaration;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.io.Section;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.comuns.collections.GrowableList;
import java.util.Iterator;

abstract class AnnotationValue extends AbstractCodeElement {

  private static final AnnotationValue EMPTY = new Empty();

  AnnotationValue() {}

  public static AnnotationValue empty() {
    return EMPTY;
  }

  public AnnotationValue addClass(ClassName type) {
    throw new IllegalArgumentException("Member cannot have different types");
  }

  public AnnotationValue addString(String string) {
    throw new IllegalArgumentException("Member cannot have different types");
  }

  private static class ClassSingle extends AnnotationValue {
    private final ClassName value;

    ClassSingle(ClassName value) {
      this.value = value;
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.writeTypeName(value).write(".class");
    }
  }

  private static class StringSingle extends AnnotationValue {
    private final String value;

    StringSingle(String value) {
      this.value = value;
    }

    @Override
    public final AnnotationValue addString(String string) {
      return new StringArray(value, string);
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      throw new UnsupportedOperationException("Implement me");
    }
  }

  private static class StringArray extends AnnotationValue {
    private final GrowableList<String> value = GrowableList.newList();

    StringArray(String v0, String v1) {
      value.add(v0);
      value.add(v1);
    }

    @Override
    public final AnnotationValue addString(String string) {
      value.add(string);
      return this;
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      w.write('{')
          .beginSection(Section.ARRAY_LITERAL);

      Iterator<String> iterator = value.iterator();
      if (iterator.hasNext()) {
        w.nextLine().writePreIndentation().writeStringLiteral(iterator.next());
        while (iterator.hasNext()) {
          w.write(',').nextLine().writePreIndentation().writeStringLiteral(iterator.next());
        }
      }

      if (!value.isEmpty()) {
        w.nextLine();
      }

      return w
          .endSection()
          .write('}');
    }
  }

  private static class Empty extends AnnotationValue {
    @Override
    public final AnnotationValue addClass(ClassName type) {
      return new ClassSingle(type);
    }

    @Override
    public final AnnotationValue addString(String string) {
      return new StringSingle(string);
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w;
    }
  }

}