/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.element;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;
import java.util.Iterator;

public abstract class AbstractImmutableCodeElement extends AbstractCodeElement {

  private final ImmutableList<CodeElement> elements;

  protected AbstractImmutableCodeElement(ImmutableList<CodeElement> elements) {
    this.elements = elements;
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w.writeCodeElements(elements);
  }

  protected static abstract class AbstractBuilder<SELF extends AbstractBuilder<SELF>> {

    private final GrowableList<CodeElement> els = GrowableList.newList();

    protected AbstractBuilder() {}

    public abstract Object build();
    
    public final SELF with(String string) {
      checkNotNull(string, "string == null");
      return withCodeElement(CodeElements.ofRaw(string));
    }

    public final SELF withCloseBrace() {
      return withCodeElement(CodeElements.CLOSE_BRACE);
    }
    
    public final SELF withCloseBracket() {
      return withCodeElement(CodeElements.CLOSE_BRACKET);
    }
    
    public final SELF withCodeElement(CodeElement element) {
      els.addWithNullMessage(element, "element == null");
      return self();
    }

    public final SELF withCommaSeparated(CodeElement... elements) {
      return withCommaSeparated(ImmutableList.newListWithAll(elements));
    }

    public final SELF withCommaSeparated(Iterable<? extends CodeElement> elements) {
      checkNotNull(elements, "elements == null");

      Iterator<? extends CodeElement> it = elements.iterator();
      if (it.hasNext()) {
        els.add(it.next());
        while (it.hasNext()) {
          els.add(CodeElements.COMMA);
          els.add(it.next());
        }
      }

      return self();
    }
    
    public final SELF withDot() {
      return withCodeElement(CodeElements.DOT);
    }

    public final SELF withIndentIfNecessary() {
      return withCodeElement(CodeElements.INDENT_IF_NECESSARY);
    }
    
    public final SELF withOpenBrace() {
      return withCodeElement(CodeElements.OPEN_BRACE);
    }
    
    public final SELF withOpenBracket() {
      return withCodeElement(CodeElements.OPEN_BRACKET);
    }

    public final SELF withSpaceOff() {
      return withCodeElement(CodeElements.SPACE_OFF);
    }
    
    public final SELF withTypeName(TypeName typeName) {
      return withCodeElement(CodeElements.ofTypeName(typeName));
    }
    
    public final SELF withWord(String word) {
      checkNotNull(word, "word == null");
      return withCodeElement(CodeElements.ofWord(word));
    }

    protected abstract SELF self();

    protected final ImmutableList<CodeElement> elements() {
      return els.toImmutableList();
    }

  }

}
