/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.statement;

import br.com.objectos.code.java.expression.LambdaBody;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.io.Section;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;

public class Block extends AbstractStatement implements LambdaBody {

  private final ImmutableList<? extends BlockElement> elements;

  private Block(ImmutableList<? extends BlockElement> elements) {
    this.elements = elements;
  }

  public static Block of(BlockElement... elements) {
    ImmutableList<BlockElement> list = ImmutableList.newListWithAll(elements);
    return new Block(list);
  }

  public static Block of(GrowableList<? extends BlockElement> elements) {
    return new Block(elements.toImmutableList());
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return elements.isEmpty()
        ? w.writeWord('{').write('}')
        : acceptCodeWriter0(w);
  }

  @Override
  public final void acceptSemicolon(Semicolon semicolon) {
    // noop
  }

  public final boolean isEmpty() {
    return elements.isEmpty();
  }

  private CodeWriter acceptCodeWriter0(CodeWriter w) {
    Semicolon semicolon = new Semicolon(w);

    w.writeWord('{');
    w.beginSection(Section.BLOCK);

    for (BlockElement statement : elements) {
      w.nextLine();
      w.beginSection(Section.STATEMENT);
      w.writeCodeElement(statement);
      w.endSection();
      statement.acceptSemicolon(semicolon);
    }

    w.nextLine();
    w.endSection();
    w.writeWord('}');
    return w;
  }

  public class Semicolon {

    private final CodeWriter w;

    private Semicolon(CodeWriter w) {
      this.w = w;
    }

    final void write() {
      w.spaceOff();
      w.write(';');
    }

  }

}
