package com.ditto.java;

import com.ditto.internal.util.InternalJavaPlatformDependencies;

import org.checkerframework.checker.nullness.qual.NonNull;

/// Factory class for creating [Ditto] instances.
///
/// This is the main entry point for creating a Ditto instance in Java applications.
/// Use [DittoConfig.Builder] to configure your Ditto instance, then pass the
/// configuration to [#create(DittoConfig)].
///
/// Example usage:
/// ```
/// DittoConfig config = new DittoConfig.Builder("my-app")
///     .persistenceDirectory("/path/to/data")
///     .build();
/// Ditto ditto = DittoFactory.create(config);
///```
///
/// @see Ditto
/// @see DittoConfig
public class DittoFactory {
    /// Private constructor to prevent instantiation.
    private DittoFactory() {
        throw new UnsupportedOperationException("DittoFactory cannot be instantiated");
    }

    /// Creates a new [Ditto] instance with the provided configuration.
    ///
    /// This method initializes a Ditto instance with Java platform-specific dependencies.
    /// The instance must be activated with a license token via [Ditto#setOfflineOnlyLicenseToken(String)]
    /// before sync can be started.
    ///
    /// @param config the [DittoConfig] configuration for the Ditto instance.
    /// @return a new [Ditto] instance configured with the provided settings.
    /// @throws NullPointerException if config is null.
    public static Ditto create(@NonNull DittoConfig config) {
        return new Ditto(config, new InternalJavaPlatformDependencies());
    }
}
