/*
 * Copyright 2017 @objectsql.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.objectsql.spring;

import com.objectsql.ObjectSQLManager;
import com.objectsql.support.CommitHandler;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import javax.sql.DataSource;

public class TxCommit {

    private PlatformTransactionManager transactionManager;
    private TransactionStatus status;
    private boolean executeSuccess = false;
    private CommitHandler commitHandler;
    private int propagation;
    private int isolationLevel;

    public TxCommit(PlatformTransactionManager manager){
        this(manager, TransactionDefinition.PROPAGATION_REQUIRED, -1);
    }

    public TxCommit(PlatformTransactionManager manager, int propagation){
        this(manager, propagation, -1);
    }

    public TxCommit(PlatformTransactionManager manager, int propagation, int isolationLevel){
        this.transactionManager = manager;
        this.propagation = propagation;
        this.isolationLevel = isolationLevel;
    }

    public TxCommit(ObjectSQLManager sqlManager){
        this(sqlManager.getDataSource(), TransactionDefinition.PROPAGATION_REQUIRED, -1);
    }

    public TxCommit(ObjectSQLManager sqlManager, int propagation){
        this(sqlManager.getDataSource(), propagation, -1);
    }

    public TxCommit(ObjectSQLManager sqlManager, int propagation, int isolationLevel){
        this(new DataSourceTransactionManager(sqlManager.getDataSource()), propagation, isolationLevel);
    }

    public TxCommit(DataSource dataSource){
        this(dataSource, TransactionDefinition.PROPAGATION_REQUIRED, -1);
    }

    public TxCommit(DataSource dataSource, int propagation){
        this(dataSource, propagation, -1);
    }

    public TxCommit(DataSource dataSource, int propagation, int isolationLevel){
        this(new DataSourceTransactionManager(dataSource), propagation, isolationLevel);
    }

    public void execute(Runnable runnable){
        execute(runnable, null);
    }
    public void execute(Runnable runnable, CommitHandler commitHandler){
        if(runnable == null){
            return;
        }
        try{
            manual();
            runnable.run();
            commit();
        }catch (Exception e){
            if(commitHandler != null){
                commitHandler.handle(e);
            }
        }finally {
            if(!executeSuccess) {
                rollback();
            }
        }
    }

    private void manual(){
        DefaultTransactionDefinition definition = new DefaultTransactionDefinition();
        definition.setPropagationBehavior(this.propagation);
        definition.setIsolationLevel(this.isolationLevel);
        status = this.transactionManager.getTransaction(definition);
        executeSuccess = false;
    }

    private void commit(){
        if(transactionManager != null && status != null ){
            transactionManager.commit(status);
            executeSuccess = true;
        }
    }

    private void rollback(){
        if(!executeSuccess && transactionManager != null && status != null ){
            transactionManager.rollback(status);
        }
    }

    public static void tx(DataSource dataSource, Runnable runnable) {
        new TxCommit(dataSource).execute(runnable);
    }

    public static void tx(DataSource dataSource, int propagation, Runnable runnable) {
        new TxCommit(dataSource, propagation).execute(runnable);
    }

    public static void tx(DataSource dataSource,int propagation, int isolationLevel, Runnable runnable) {
        new TxCommit(dataSource, propagation, isolationLevel).execute(runnable);
    }

    public static void tx(DataSource dataSource, Runnable runnable, CommitHandler commitHandler) {
        new TxCommit(dataSource).execute(runnable, commitHandler);
    }

    public static void tx(DataSource dataSource, int propagation, Runnable runnable, CommitHandler commitHandler) {
        new TxCommit(dataSource, propagation).execute(runnable, commitHandler);
    }

    public static void tx(DataSource dataSource, int propagation, int isolationLevel, Runnable runnable, CommitHandler commitHandler) {
        new TxCommit(dataSource, propagation, isolationLevel).execute(runnable, commitHandler);
    }

    public static void tx(ObjectSQLManager objectSQLManager, Runnable runnable) {
        new TxCommit(objectSQLManager).execute(runnable);
    }

    public static void tx(ObjectSQLManager objectSQLManager, int propagation, Runnable runnable) {
        new TxCommit(objectSQLManager, propagation).execute(runnable);
    }

    public static void tx(ObjectSQLManager objectSQLManager,int propagation, int isolationLevel, Runnable runnable) {
        new TxCommit(objectSQLManager, propagation, isolationLevel).execute(runnable);
    }

    public static void tx(ObjectSQLManager objectSQLManager, Runnable runnable, CommitHandler commitHandler) {
        new TxCommit(objectSQLManager).execute(runnable, commitHandler);
    }

    public static void tx(ObjectSQLManager objectSQLManager, int propagation, Runnable runnable, CommitHandler commitHandler) {
        new TxCommit(objectSQLManager, propagation).execute(runnable, commitHandler);
    }

    public static void tx(ObjectSQLManager objectSQLManager, int propagation, int isolationLevel, Runnable runnable, CommitHandler commitHandler) {
        new TxCommit(objectSQLManager, propagation, isolationLevel).execute(runnable, commitHandler);
    }

    public static void tx(DataSourceTransactionManager dataSourceTransactionManager, Runnable runnable) {
        new TxCommit(dataSourceTransactionManager).execute(runnable);
    }

    public static void tx(DataSourceTransactionManager dataSourceTransactionManager, int propagation, Runnable runnable) {
        new TxCommit(dataSourceTransactionManager, propagation).execute(runnable);
    }

    public static void tx(DataSourceTransactionManager dataSourceTransactionManager,int propagation, int isolationLevel, Runnable runnable) {
        new TxCommit(dataSourceTransactionManager, propagation, isolationLevel).execute(runnable);
    }

    public static void tx(DataSourceTransactionManager dataSourceTransactionManager, Runnable runnable, CommitHandler commitHandler) {
        new TxCommit(dataSourceTransactionManager).execute(runnable, commitHandler);
    }

    public static void tx(DataSourceTransactionManager dataSourceTransactionManager, int propagation, Runnable runnable, CommitHandler commitHandler) {
        new TxCommit(dataSourceTransactionManager, propagation).execute(runnable, commitHandler);
    }

    public static void tx(DataSourceTransactionManager dataSourceTransactionManager, int propagation, int isolationLevel, Runnable runnable, CommitHandler commitHandler) {
        new TxCommit(dataSourceTransactionManager, propagation, isolationLevel).execute(runnable, commitHandler);
    }

}
