/* 
 * Code generated by Speakeasy (https://speakeasy.com). DO NOT EDIT.
 */
package com.stackone.stackone_client_java.utils;

import java.io.IOException;
import java.net.ConnectException;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.*;
import java.util.function.Supplier;
import com.stackone.stackone_client_java.utils.Blob;

public class AsyncRetries {

    private final RetryConfig retryConfig;
    private final List<String> retriableStatusCodes;
    private final ScheduledExecutorService scheduler;

    private AsyncRetries(RetryConfig retryConfig,
                         List<String> retriableStatusCodes,
                         ScheduledExecutorService scheduler) {
        Utils.checkNotNull(retryConfig, "retryConfig");
        Utils.checkNotNull(retriableStatusCodes, "statusCodes");
        if (retriableStatusCodes.isEmpty()) {
            throw new IllegalArgumentException("statusCodes list cannot be empty");
        }
        this.retryConfig = retryConfig;
        this.retriableStatusCodes = retriableStatusCodes;
        this.scheduler = scheduler;
    }

    public CompletableFuture<HttpResponse<Blob>> retry(
            Supplier<CompletableFuture<HttpResponse<Blob>>> task
    ) {
        switch (retryConfig.strategy()) {
            case BACKOFF:
                CompletableFuture<HttpResponse<Blob>> future = new CompletableFuture<>();
                BackoffStrategy backoff = retryConfig.backoff()
                        // We want to fail fast during misconfigurations.
                        .orElseThrow(() -> new IllegalArgumentException("Backoff strategy is not defined"));
                attempt(task, future, backoff, new State(0, Instant.now()));
                return future;
            case NONE:
                return task.get();
            default:
                throw new IllegalArgumentException("Unsupported retry strategy: " + retryConfig.strategy());
        }
    }

    private <T> void attempt(Supplier<CompletableFuture<HttpResponse<Blob>>> task,
                             CompletableFuture<HttpResponse<Blob>> result,
                             BackoffStrategy backoff,
                             State state) {
        task.get().whenComplete((response, throwable) -> {
            if (throwable == null) {
                boolean matched = retriableStatusCodes.stream()
                        .anyMatch(pattern -> Utils.statusCodeMatches(response.statusCode(), pattern));
                if (matched) {
                    maybeRetry(task, result, backoff, state, new AsyncRetryableException(response));
                    return;
                }
                result.complete(response);
                return;
            }
            // Unwrap
            Throwable e = (throwable instanceof CompletionException) ? throwable.getCause() : throwable;
            if (e instanceof AsyncRetryableException) {
                maybeRetry(task, result, backoff, state, e);
                return;
            }
            if (e instanceof IOException) {
                if (shouldRetryIOException(e, backoff)) {
                    maybeRetry(task, result, backoff, state, e);
                    return;
                }
            }
            result.completeExceptionally(new NonRetryableException(e));
        });
    }

    private boolean shouldRetryIOException(Throwable e, BackoffStrategy backoff) {
        if (e instanceof ConnectException && backoff.retryConnectError()) return true;
        String message = e.getMessage();
        if (message == null) return false;
        return (message.contains("Connect timed out") && backoff.retryConnectError())
                || (message.contains("Read timed out") && backoff.retryReadTimeoutError());
    }

    private void maybeRetry(Supplier<CompletableFuture<HttpResponse<Blob>>> task,
                            CompletableFuture<HttpResponse<Blob>> result,
                            BackoffStrategy backoff,
                            State state,
                            Throwable e) {
        Duration timeSinceStart = Duration.between(state.startedAt(), Instant.now());
        if (timeSinceStart.toMillis() > backoff.maxElapsedTimeMs()) {
            // retry exhausted
            if (e instanceof AsyncRetryableException) {
                result.complete(((AsyncRetryableException) e).response());
                return;
            }
            result.completeExceptionally(e);
            return;
        }

        double intervalMs = backoff.initialIntervalMs() * Math.pow(backoff.baseFactor(), state.count());
        double jitterMs = backoff.jitterFactor() * intervalMs;
        intervalMs = intervalMs - jitterMs + Math.random() * (2 * jitterMs + 1);
        intervalMs = Math.min(intervalMs, backoff.maxIntervalMs());

        scheduler.schedule(
                () -> attempt(task, result, backoff, state.countAttempt()),
                (long) intervalMs,
                TimeUnit.MILLISECONDS);
    }

    public void shutdown() {
        scheduler.shutdown();
    }

    public static Builder builder() {
        return new Builder();
    }

    public final static class Builder {

        private RetryConfig retryConfig;
        private List<String> statusCodes;
        private ScheduledExecutorService scheduler;

        private Builder() {
        }

        /**
         * Defines the retry configuration.
         *
         * @param retryConfig The retry configuration to use.
         * @return The builder instance.
         */
        public Builder retryConfig(RetryConfig retryConfig) {
            Utils.checkNotNull(retryConfig, "retryConfig");
            this.retryConfig = retryConfig;
            return this;
        }

        /**
         * Defines the status codes that should be considered as errors.
         *
         * @param statusCodes The list of status codes to treat as errors.
         * @return The builder instance.
         */
        public Builder statusCodes(List<String> statusCodes) {
            Utils.checkNotNull(statusCodes, "statusCodes");
            if (statusCodes.isEmpty()) {
                throw new IllegalArgumentException("statusCodes list cannot be empty");
            }
            this.statusCodes = statusCodes;
            return this;
        }

        /**
         * Defines the scheduler that will be used to schedule and execute retry attempts.
         * Recommend using a globally shared executor for this.
         *
         * @param scheduler An instance of {@link ScheduledExecutorService}
         * @return The builder instance.
         */
        public Builder scheduler(ScheduledExecutorService scheduler) {
            Utils.checkNotNull(scheduler, "scheduler");
            this.scheduler = scheduler;
            return this;
        }

        public AsyncRetries build() {
            return new AsyncRetries(retryConfig, statusCodes, scheduler);
        }
    }

    private static class State {
        private long attempt;
        private final Instant startedAt;

        public State(long attempt, Instant startedAt) {
            this.attempt = attempt;
            this.startedAt = startedAt;
        }

        public long count() {
            return attempt;
        }

        public Instant startedAt() {
            return startedAt;
        }

        public State countAttempt() {
            attempt++;
            return this;
        }
    }

}
