/* 
 * Code generated by Speakeasy (https://speakeasy.com). DO NOT EDIT.
 */
package com.stackone.stackone_client_java.utils;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Optional;

/**
 * Generic streaming parser that handles byte buffer management and delegates
 * format-specific logic to a StreamContentProcessor.
 */
public final class StreamingParser<T> {

    /**
    * Information about a found boundary in byte data
    */
    static class BoundaryInfo {
        public final int position;
        public final int delimiterLength;

        public BoundaryInfo(int position, int delimiterLength) {
            this.position = position;
            this.delimiterLength = delimiterLength;
        }
    }

    /**
    * Interface for format-specific parsing logic
    */
    interface StreamContentProcessor<T> {
        /**
        * Find the next boundary in the byte buffer
        * @return boundary info, or position -1 if no boundary found
        */
        BoundaryInfo findBoundary(byte[] data, int limit);

        /**
        * Process extracted content and return the parsed result
        * @param content the extracted content (without boundary delimiters)
        * @return parsed result, or empty if content should be skipped
        */
        Optional<T> processContent(String content);

        /**
        * Sanitize content text (e.g., handle line endings, BOM, etc.)
        * @param rawContent the raw extracted content
        * @param isFirst whether this is the first content processed
        * @return sanitized content
        */
        default String sanitizeContent(String rawContent, boolean isFirst) {
            return rawContent.replace("\r\n", "\n").replace("\r", "\n");
        }
    }

    private final StreamContentProcessor<T> processor;
    private ByteBuffer byteBuffer = ByteBuffer.allocate(8192);
    private boolean first = true;

    StreamingParser(StreamContentProcessor<T> processor) {
        this.processor = processor;
    }

    /**
     * Add ByteBuffer data to the parser buffer and extract any complete items.
     *
     * @param inputBuffer byte data to add (will not be modified)
     * @return next complete parsed result if one becomes available
     */
    public Optional<T> add(ByteBuffer inputBuffer) {
        if (inputBuffer == null || !inputBuffer.hasRemaining()) {
            return extractNextFromBytes();
        }
        // Ensure we have enough capacity
        if (byteBuffer.remaining() < inputBuffer.remaining()) {
            byteBuffer = expandByteBuffer(byteBuffer.position() + inputBuffer.remaining());
        }
        // Append new data
        byteBuffer.put(inputBuffer.slice());
        return extractNextFromBytes();
    }

    /**
     * Extract any remaining partial content when stream ends.
     *
     * @return final parsed result if there was incomplete data in the buffer
     */
    public Optional<T> finish() {
        if (byteBuffer.position() > 0) {
            byte[] remainingBytes = new byte[byteBuffer.position()];
            byteBuffer.flip();
            byteBuffer.get(remainingBytes);
            byteBuffer.clear();
            String content = processor.sanitizeContent(new String(remainingBytes, StandardCharsets.UTF_8), first);
            return processor.processContent(content);
        }
        return Optional.empty();
    }

    /**
     * Check if there are additional complete items in the buffer.
     *
     * @return next complete parsed result from buffer if available
     */
    public Optional<T> next() {
        return extractNextFromBytes();
    }

    /**
     * Check if parser has any buffered data
     */
    public boolean hasBufferedData() {
        return byteBuffer.position() > 0;
    }

    private Optional<T> extractNextFromBytes() {
        if (byteBuffer.position() == 0) {
            return Optional.empty();
        }
        // Find boundary directly in bytes
        BoundaryInfo boundary = processor.findBoundary(byteBuffer.array(), byteBuffer.position());
        if (boundary.position == -1) {
            return Optional.empty();
        }
        // Extract content bytes without copying the entire buffer
        byte[] contentBytes = new byte[boundary.position];
        byteBuffer.flip();
        byteBuffer.get(contentBytes, 0, boundary.position);
        // Compact buffer to remove processed content + delimiter
        byteBuffer.position(boundary.position + boundary.delimiterLength);
        byteBuffer.compact();
        String content = processor.sanitizeContent(new String(contentBytes, StandardCharsets.UTF_8), first);
        if (first) {
            first = false;
        }
        Optional<T> result = processor.processContent(content);
        if (result.isPresent()) {
            return result;
        }
        // Check for additional items if this one was skipped
        return extractNextFromBytes();
    }

    private ByteBuffer expandByteBuffer(int newCapacity) {
        ByteBuffer newBuffer = ByteBuffer.allocate(Math.max(newCapacity, byteBuffer.capacity() * 2));
        byteBuffer.flip();
        newBuffer.put(byteBuffer);
        return newBuffer;
    }

    /**
     * Check if a byte pattern matches at a specific position
     */
    public static boolean matchesPattern(byte[] data, int pos, int limit, byte... pattern) {
        if (pos + pattern.length > limit) {
            return false;
        }
        for (int i = 0; i < pattern.length; i++) {
            if (data[pos + i] != pattern[i]) {
                return false;
            }
        }
        return true;
    }

    // ===== JSON Lines Content Processor =====
    
    /**
     * JSON Lines content processor implementation
     */
    private static class JsonLContentProcessor implements StreamContentProcessor<String> {
        // Line boundary patterns
        private static final byte CR = '\r';
        private static final byte LF = '\n';
        private static final byte[] CRLF = {CR, LF}; // \r\n
        private static final byte[] LF_ONLY = {LF}; // \n

        @Override
        public BoundaryInfo findBoundary(byte[] data, int limit) {
            for (int i = 0; i < limit; i++) {
                // Check for CRLF first (longer pattern)
                if (matchesPattern(data, i, limit, CRLF)) {
                    return new BoundaryInfo(i, CRLF.length);
                }
                // Check for LF only
                if (matchesPattern(data, i, limit, LF_ONLY)) {
                    return new BoundaryInfo(i, LF_ONLY.length);
                }
            }
            return new BoundaryInfo(-1, 0);
        }

        @Override
        public Optional<String> processContent(String content) {
            String trimmed = content.trim();
            // Return non-empty JSON lines
            return trimmed.isEmpty() ? Optional.empty() : Optional.of(trimmed);
        }
    }

    // ===== SSE Content Processor =====
    
    /**
     * SSE content processor implementation
     */
    private static class SSEContentProcessor implements StreamContentProcessor<EventStreamMessage> {
        private static final String BYTE_ORDER_MARK = "\uFEFF";
        private static final char LINEFEED = '\n';
        private static final byte CR = '\r';
        private static final byte LF = '\n';
        private static final byte[][] BOUNDARY_PATTERNS = {
            {CR, LF},
            {LF},
            {CR}
        };

        private Optional<String> eventId = Optional.empty();

        @Override
        public BoundaryInfo findBoundary(byte[] data, int limit) {
            int lineStart = 0, i = lineStart;
            while (i < limit) {
                for (byte[] pattern : BOUNDARY_PATTERNS) {
                    if (matchesPattern(data, i, limit, pattern)) {
                        if (i == lineStart) { // empty line
                            int boundStart = i;
                            while (boundStart > 0 && (data[boundStart - 1] == CR || data[boundStart - 1] == LF)) {
                                boundStart--;
                            }
                            int boundLength = (lineStart - boundStart) + pattern.length;
                            return new BoundaryInfo(boundStart, boundLength);
                        }
                        lineStart = i + pattern.length;
                        i = lineStart - 1;
                        break;
                    }
                }
                i++;
            }
            return new BoundaryInfo(-1, 0);
        }

        @Override
        public Optional<EventStreamMessage> processContent(String content) {
            if (content.trim().isEmpty()) {
                return Optional.empty();
            }
            return Optional.of(parseMessage(content));
        }

        @Override
        public String sanitizeContent(String rawContent, boolean isFirst) {
            String sanitized = rawContent.replace("\r\n", "\n").replace("\r", "\n");
            if (isFirst && sanitized.startsWith(BYTE_ORDER_MARK)) {
                sanitized = sanitized.substring(BYTE_ORDER_MARK.length());
            }
            return sanitized;
        }

        private EventStreamMessage parseMessage(String text) {
            String[] lines = text.split("\n");
            Optional<String> event = Optional.empty();
            Optional<Integer> retryMs = Optional.empty();
            Optional<StringBuilder> data = Optional.empty();
            for (String line : lines) {
                if (line.startsWith(":")) {
                    continue;
                }
                String key;
                String value;
                int colonIndex = line.indexOf(':');
                if (colonIndex >= 0) {
                    key = line.substring(0, colonIndex);
                    value = line.substring(colonIndex + 1);
                    if (value.startsWith(" ")) {
                        value = value.substring(1);
                    }
                } else {
                    key = line;
                    value = "";
                }
                switch (key) {
                    case "event":
                        event = Optional.of(value);
                        break;
                    case "id":
                        if (value.indexOf('\0') < 0) {
                            eventId = Optional.of(value);
                        }
                        break;
                    case "retry":
                        try {
                            retryMs = Optional.of(Integer.parseInt(value));
                        } catch (NumberFormatException e) {
                            // ignore invalid retry values
                        }
                        break;
                    case "data":
                        if (data.isEmpty()) {
                            data = Optional.of(new StringBuilder());
                        } else {
                            data.get().append(LINEFEED);
                        }
                        data.get().append(value);
                        break;
                }
            }
            return new EventStreamMessage(event, eventId, retryMs, data.map(StringBuilder::toString));
        }
    }

    // ===== Factory Methods =====

    /**
     * Create a streaming parser for JSON Lines format
     */
    public static StreamingParser<String> forJsonLines() {
        return new StreamingParser<>(new JsonLContentProcessor());
    }

    /**
     * Create a streaming parser for SSE format
     */
    public static StreamingParser<EventStreamMessage> forSSE() {
        return new StreamingParser<>(new SSEContentProcessor());
    }
}
