package fi.evolver.script.app;

import java.io.IOException;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Map;
import java.util.stream.Stream;

import fi.evolver.script.Download;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;
import fi.evolver.script.system.Architecture;

public class Eclipse {
    private static final String INSTALL_URL_TEMPLATE = "https://gemmei.ftp.acc.umu.se/mirror/eclipse.org/technology/epp/downloads/release/%1$s/R/eclipse-jee-%1$s-R-linux-gtk-%2$s.tar.gz";
    private static final Map<Architecture, String> ARCH_SUFFIXES = Map.of(
            Architecture.AMD64, "x86_64",
            Architecture.ARM64, "aarch64");

    private static final Path APPS_DIR = Shell.HOME.resolve("apps");
	private static final Path INSTALL_DIR = APPS_DIR.resolve("eclipse");


    /**
	 * Installs the Eclipse JEE version.
	 *
	 * <p>If Eclipse is already installed in the target directory, the installation is skipped.
	 * Otherwise, the distribution archive is downloaded, extracted, and the resulting folder is renamed to a
	 * standardized installation folder.</p>
     *
	 * @param version Eclipse version, such as "2025-06"
     * @throws IOException if downloading the archive fails. 
	 */
    public static void install(String version) throws IOException {
        try (Step step = Step.start("Eclipse %s: install".formatted(version))) {
			if (Files.exists(INSTALL_DIR)) {
				step.skip("Eclipse is already installed");
				return;
			}
			Files.createDirectories(APPS_DIR);

            URI downloadUri = URI.create(INSTALL_URL_TEMPLATE.formatted(version, ARCH_SUFFIXES.get(Architecture.validated())));
            Path downloadedTarGz = Download.intoTempFile(downloadUri);

			Shell.user("tar", "-xzf", downloadedTarGz.toString(), "-C", APPS_DIR.toString());

			try (Stream<Path> dirStream = Files.list(APPS_DIR)) {
				Path extractedDir = dirStream
						.filter(p -> Files.isDirectory(p) && p.getFileName().toString().startsWith("eclipse"))
						.findFirst()
						.orElseThrow(() -> new RuntimeException("Extracted Eclipse folder not found"));

				Files.move(extractedDir, INSTALL_DIR);
			}
        }
    }

}
