package org.sqlproc.engine.impl.type;

import java.lang.reflect.Method;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.type.Type;
import org.joda.time.DateTime;
import org.joda.time.LocalTime;
import org.sqlproc.engine.SqlRuntimeException;
import org.sqlproc.engine.impl.BeanUtils;

/**
 * The SQL META Types for TIME.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public class SqlTimeType extends SqlGenericType {

    /**
     * {@inheritDoc}
     */
    @Override
    public Type getHibernateType() {
        return Hibernate.TIME;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getClassTypes() {
        return new Class[] {};
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getMetaTypes() {
        return new String[] { "TIME" };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setResult(Object resultInstance, String attributeName, Object resultValue, boolean ingoreError)
            throws SqlRuntimeException {
        if (logger.isTraceEnabled()) {
            logger.trace(">>> setResult " + getMetaTypes()[0] + ": resultInstance=" + resultInstance
                    + ", attributeName=" + attributeName + ", resultValue=" + resultValue);
        }
        Method m = BeanUtils.getSetter(resultInstance, attributeName, java.util.Date.class);
        if (m != null) {
            BeanUtils.simpleInvokeMethod(m, resultInstance, resultValue);
        } else {
            m = BeanUtils.getSetter(resultInstance, attributeName, LocalTime.class);
            if (m != null) {
                LocalTime dt = new LocalTime(resultValue);
                BeanUtils.simpleInvokeMethod(m, resultInstance, dt);
            } else if (ingoreError) {
                logger.error("There's no getter for " + attributeName + " in " + resultInstance + ", META type is "
                        + getMetaTypes()[0]);
            } else {
                throw new SqlRuntimeException("There's no setter for " + attributeName + " in " + resultInstance
                        + ", META type is " + getMetaTypes()[0]);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setParameter(Query query, String paramName, Object inputValue, Class<?> inputType, boolean ingoreError)
            throws SqlRuntimeException {
        if (logger.isTraceEnabled()) {
            logger.trace(">>> setParameter " + getMetaTypes()[0] + ": paramName=" + paramName + ", inputValue="
                    + inputValue + ", inputType=" + inputType);
        }
        if (inputValue == null) {
            query.setParameter(paramName, inputValue, Hibernate.TIME);
        } else {
            DateTime dt = new DateTime(inputValue).withMillisOfSecond(0);
            query.setTime(paramName, dt.toDate());
        }
    }
}
