package org.sqlproc.engine.impl.type;

import java.lang.reflect.Method;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.type.Type;
import org.sqlproc.engine.SqlRuntimeException;
import org.sqlproc.engine.impl.BeanUtils;

/**
 * The SQL META Types for TIMESTAMP.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public class SqlTimestampType extends SqlGenericType {

    /**
     * {@inheritDoc}
     */
    @Override
    public Type getHibernateType() {
        return Hibernate.TIMESTAMP;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getClassTypes() {
        return new Class[] { java.sql.Timestamp.class };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getMetaTypes() {
        return new String[] { "TIMESTAMP", "STAMP" };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setResult(Object resultInstance, String attributeName, Object resultValue, boolean ingoreError)
            throws SqlRuntimeException {
        if (logger.isTraceEnabled()) {
            logger.trace(">>> setResult " + getMetaTypes()[0] + ": resultInstance=" + resultInstance
                    + ", attributeName=" + attributeName + ", resultValue=" + resultValue);
        }
        Method m = BeanUtils.getSetter(resultInstance, attributeName, java.sql.Timestamp.class, java.util.Date.class);
        if (m != null) {
            BeanUtils.simpleInvokeMethod(m, resultInstance, resultValue);
        } else if (ingoreError) {
            logger.error("There's no getter for " + attributeName + " in " + resultInstance + ", META type is "
                    + getMetaTypes()[0]);
        } else {
            throw new SqlRuntimeException("There's no setter for " + attributeName + " in " + resultInstance
                    + ", META type is " + getMetaTypes()[0]);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setParameter(Query query, String paramName, Object inputValue, Class<?> inputType, boolean ingoreError)
            throws SqlRuntimeException {
        if (logger.isTraceEnabled()) {
            logger.trace(">>> setParameter " + getMetaTypes()[0] + ": paramName=" + paramName + ", inputValue="
                    + inputValue + ", inputType=" + inputType);
        }
        if (inputValue == null) {
            query.setParameter(paramName, inputValue, Hibernate.TIMESTAMP);
        } else {
            if (inputValue instanceof java.sql.Timestamp) {
                query.setTimestamp(paramName, (java.sql.Timestamp) inputValue);
            } else if (inputValue instanceof java.util.Date) {
                query.setTimestamp(paramName, (java.util.Date) inputValue);
            } else if (ingoreError) {
                logger.error("Incorrect timestamp " + inputValue + " for " + paramName);
            } else {
                throw new SqlRuntimeException("Incorrect timestamp " + inputValue + " for " + paramName);
            }
        }
    }
}
