/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.declaration;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.io.Section;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.comuns.collections.GrowableList;
import java.util.Iterator;

abstract class AnnotationValue extends AbstractCodeElement {

  private static final AnnotationValue EMPTY = new Empty();

  AnnotationValue() {}

  public static AnnotationValue empty() {
    return EMPTY;
  }

  public AnnotationValue addClass(ClassName type) {
    throw new IllegalArgumentException("Member cannot have different types");
  }

  public AnnotationValue addString(String string) {
    throw new IllegalArgumentException("Member cannot have different types");
  }

  private static class ClassSingle extends AnnotationValue {
    private final ClassName value;

    ClassSingle(ClassName value) {
      this.value = value;
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.writeTypeName(value).write(".class");
    }
  }

  private static class StringSingle extends AnnotationValue {
    private final String value;

    StringSingle(String value) {
      this.value = value;
    }

    @Override
    public final AnnotationValue addString(String string) {
      return new StringArray(value, string);
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      throw new UnsupportedOperationException("Implement me");
    }
  }

  private static class StringArray extends AnnotationValue {
    private final GrowableList<String> value = GrowableList.newList();

    StringArray(String v0, String v1) {
      value.add(v0);
      value.add(v1);
    }

    @Override
    public final AnnotationValue addString(String string) {
      value.add(string);
      return this;
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      w.write('{')
          .beginSection(Section.ARRAY_LITERAL);

      Iterator<String> iterator = value.iterator();
      if (iterator.hasNext()) {
        w.nextLine().writePreIndentation().writeStringLiteral(iterator.next());
        while (iterator.hasNext()) {
          w.write(',').nextLine().writePreIndentation().writeStringLiteral(iterator.next());
        }
      }

      if (!value.isEmpty()) {
        w.nextLine();
      }

      return w
          .endSection()
          .write('}');
    }
  }

  private static class Empty extends AnnotationValue {
    @Override
    public final AnnotationValue addClass(ClassName type) {
      return new ClassSingle(type);
    }

    @Override
    public final AnnotationValue addString(String string) {
      return new StringSingle(string);
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w;
    }
  }

}