/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.io.BodyFormatter;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.ClassNameOrParameterizedTypeName;
import br.com.objectos.code.java.type.TypeParameterName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;
import br.com.objectos.comuns.collections.StreamIterable;
import java.lang.annotation.Annotation;
import javax.lang.model.element.TypeElement;

public class ClassCode extends TypeCode {

  private final ImmutableList<AnnotatedCode> annotationList;
  private final ClassModifier modifier;
  private final String simpleName;
  private final ImmutableList<TypeParameterName> typeParameters;
  private final Extends _extends;
  private final ImmutableList<ClassNameOrParameterizedTypeName> interfaces;
  private final ImmutableList<ClassBodyElement> bodyElements;

  private ClassCode(Builder builder) {
    annotationList = builder.annotationList();
    modifier = builder.modifier();
    simpleName = builder.simpleName;
    typeParameters = builder.typeParameters();
    _extends = builder._extends;
    interfaces = builder.interfaces();
    bodyElements = builder.bodyElements();
  }

  public static Builder _class(ClassName className) {
    return builder()._class(className);
  }

  public static Builder _class(String simpleName) {
    return builder()._class(simpleName);
  }

  public static Builder _final() {
    return builder()._final();
  }

  public static Builder _private() {
    return builder()._private();
  }

  public static Builder _public() {
    return builder()._public();
  }

  public static Builder annotatedWith(AnnotatedCode annotation) {
    return builder().annotatedWith(annotation);
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w
        .pushSimpleName(simpleName)
        .writeAnnotations(annotationList)
        .writeCodeElement(modifier)
        .writeWord("class")
        .writeSimpleNameWith(typeParameters)
        .writeCodeElement(_extends)
        .writeImplementsIfNecessary(interfaces)
        .writeBlock(bodyElements)
        .popSimpleName();
  }

  @Override
  public final String simpleName() {
    return simpleName;
  }

  public static class Builder {

    private final GrowableList<AnnotatedCode> annotatedList = GrowableList.newList();
    private final ClassModifier.Builder modifier = ClassModifier.builder();
    private String simpleName = "Unnamed";
    private final GrowableList<TypeParameterName> typeParameters = GrowableList.newList();
    private Extends _extends = Extends.none();
    private final GrowableList<ClassNameOrParameterizedTypeName> interfaces
        = GrowableList.newList();
    private final GrowableList<ClassBodyElement> bodyElements = GrowableList.newList();

    private BodyFormatter formatter = BodyFormatter.defaultFormatter();

    private Builder() {}

    public final Builder _class(ClassName className) {
      checkNotNull(className, "className == null");
      return _class(className.simpleName());
    }

    public final Builder _class(String simpleName) {
      this.simpleName = checkNotNull(simpleName, "simpleName == null");
      return this;
    }

    public final Builder _extends(ClassNameOrParameterizedTypeName superclass) {
      checkNotNull(superclass, "superclass == null");
      _extends = Extends.ofTypeName(superclass);
      return this;
    }

    public final Builder _implements(Iterable<? extends ClassNameOrParameterizedTypeName> ifaces) {
      checkNotNull(ifaces, "ifaces == null");
      interfaces.addAll(ifaces);
      return this;
    }

    public final Builder _abstract() {
      modifier._abstract();
      return this;
    }

    public final Builder _final() {
      modifier._final();
      return this;
    }

    public final Builder _private() {
      modifier._private();
      return this;
    }

    public final Builder _public() {
      modifier._public();
      return this;
    }

    public final Builder _static() {
      modifier._static();
      return this;
    }

    public final Builder annotatedWith(AnnotatedCode annotated) {
      annotatedList.add(annotated);
      return this;
    }

    public final Builder annotatedWith(Class<? extends Annotation> annotationType) {
      return annotatedWith(AnnotatedCode.annotatedWith(annotationType));
    }

    public final Builder body(ClassBodyElement... elements) {
      bodyElements.fluentAdd(elements);
      return this;
    }

    public final Builder body(Iterable<? extends ClassBodyElement> elements) {
      return withBodyElements(elements);
    }

    public final ClassCode build() {
      return new ClassCode(this);
    }

    public final ClassCode buildWith(BodyFormatter formatter) {
      this.formatter = checkNotNull(formatter, "formatter == null");
      return build();
    }

    public final Builder withBodyElement(ClassBodyElement element) {
      bodyElements.add(element);
      return this;
    }

    public final Builder withBodyElements(Iterable<? extends ClassBodyElement> elements) {
      bodyElements.addAll(elements);
      return this;
    }

    public final Builder withModifier(ClassModifier modifier) {
      this.modifier.withModifier(modifier);
      return this;
    }

    public final Builder withInterface(ClassNameOrParameterizedTypeName iface) {
      interfaces.addWithNullMessage(iface, "iface == null");
      return this;
    }

    public final Builder withTypeParameter(String name) {
      return withTypeParameter0(TypeParameterName.named(name));
    }

    public final Builder withTypeParameter(TypeParameterName parameter) {
      checkNotNull(parameter, "parameter == null");
      return withTypeParameter0(parameter);
    }

    public final Builder withTypeParameters(Iterable<TypeParameterName> parameters) {
      checkNotNull(parameters, "parameters == null");
      return withTypeParameters0(parameters);
    }

    public final Builder withTypeParametersFrom(TypeElement type) {
      checkNotNull(type, "type == null");
      StreamIterable<TypeParameterName> parameters = TypeParameterName.streamIterableOf(type);
      return withTypeParameters0(parameters);
    }

    final ImmutableList<AnnotatedCode> annotationList() {
      return annotatedList.toImmutableList();
    }

    final ImmutableList<ClassBodyElement> bodyElements() {
      return formatter.format(bodyElements, ClassBodyElement.class);
    }

    final ImmutableList<ClassNameOrParameterizedTypeName> interfaces() {
      return interfaces.toImmutableList();
    }

    final ClassModifier modifier() {
      return modifier.build();
    }

    final ImmutableList<TypeParameterName> typeParameters() {
      return typeParameters.toImmutableList();
    }

    private Builder withTypeParameter0(TypeParameterName name) {
      typeParameters.add(name);
      return this;
    }

    private Builder withTypeParameters0(Iterable<TypeParameterName> parameters) {
      typeParameters.addAll(parameters);
      return this;
    }

  }

}