/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.expression.LambdaParameter;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.TypeName;
import java.util.Objects;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeMirror;

public class ParameterCode extends AbstractCodeElement
    implements
    LambdaParameter {

  private final TypeName typeName;
  private final String name;

  ParameterCode(TypeName typeName, String name) {
    this.typeName = typeName;
    this.name = name;
  }

  public static ParameterCode of(VariableElement element) {
    checkNotNull(element, "element == null");
    TypeMirror type = element.asType();
    return new ParameterCode(
        TypeName.ofUnchecked(type),
        element.getSimpleName().toString());
  }

  public static ParameterCode of(Class<?> type, String name) {
    ClassName typeName = ClassName.of(type);
    checkNotNull(name, "name == null");
    return new ParameterCode(typeName, name);
  }

  public static ParameterCode of(TypeName typeName, String name) {
    checkNotNull(typeName, "typeName == null");
    checkNotNull(name, "name == null");
    return new ParameterCode(typeName, name);
  }

  static ParameterCode ofUnchecked(TypeName typeName, String name) {
    return new ParameterCode(typeName, name);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter writer) {
    return writer.writeTypeNameAsWord(typeName).writeWord(name);
  }

  @Override
  public final boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (!(obj instanceof ParameterCode)) {
      return false;
    }
    ParameterCode that = (ParameterCode) obj;
    return Objects.equals(typeName, that.typeName)
        && Objects.equals(name, that.name);
  }

  @Override
  public final int hashCode() {
    return Objects.hash(typeName, name);
  }

  public final String name() {
    return name;
  }

  public final TypeName typeName() {
    return typeName;
  }

}