/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.io.ImportSet;
import br.com.objectos.code.java.io.JavaFile;
import br.com.objectos.code.java.type.ClassName;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.io.Writer;
import javax.annotation.processing.Filer;
import javax.tools.JavaFileObject;

abstract class TypeCode extends AbstractCodeElement implements ClassBodyElement {

  TypeCode() {}

  @Override
  public final Kind kind() {
    return Kind.TYPE;
  }

  public abstract String simpleName();

  public final JavaFile toJavaFile(PackageName packageName) {
    checkNotNull(packageName, "packageName == null");
    return new TypeCodeJavaFile(packageName, this);
  }

  private static class TypeCodeJavaFile implements JavaFile {

    private final PackageName packageName;
    private final TypeCode typeCode;

    TypeCodeJavaFile(PackageName packageName, TypeCode typeCode) {
      this.packageName = packageName;
      this.typeCode = typeCode;
    }

    @Override
    public final String toString() {
      ImportSet importSet = ImportSet.forPackageName(packageName);
      CodeWriter writer = CodeWriter.forJavaFile(importSet);
      typeCode.acceptCodeWriter(writer);
      return writer.toJavaFile();
    }

    @Override
    public final void writeTo(Filer filer) {
      try {
        writeTo0(filer);
      } catch (IOException e) {
        throw new UncheckedIOException(e);
      }
    }

    private void writeTo0(Filer filer) throws IOException {
      ClassName className = packageName.nestedClass(typeCode.simpleName());
      JavaFileObject object = filer.createSourceFile(className.toString());
      try (Writer writer = object.openWriter()) {
        writer.write(toString());
      }
    }

  }

}