/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.element;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;
import java.util.Iterator;

public abstract class AbstractImmutableCodeElement extends AbstractCodeElement {

  private final ImmutableList<CodeElement> elements;

  protected AbstractImmutableCodeElement(ImmutableList<CodeElement> elements) {
    this.elements = elements;
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w.writeCodeElements(elements);
  }

  protected static abstract class AbstractBuilder<SELF extends AbstractBuilder<SELF>> {

    private final GrowableList<CodeElement> els = GrowableList.newList();

    protected AbstractBuilder() {}

    public abstract Object build();
    
    public final SELF with(String string) {
      checkNotNull(string, "string == null");
      return withCodeElement(CodeElements.ofRaw(string));
    }

    public final SELF withCloseBrace() {
      return withCodeElement(CodeElements.CLOSE_BRACE);
    }
    
    public final SELF withCloseBracket() {
      return withCodeElement(CodeElements.CLOSE_BRACKET);
    }
    
    public final SELF withCodeElement(CodeElement element) {
      els.addWithNullMessage(element, "element == null");
      return self();
    }

    public final SELF withCommaSeparated(CodeElement... elements) {
      return withCommaSeparated(ImmutableList.newListWithAll(elements));
    }

    public final SELF withCommaSeparated(Iterable<? extends CodeElement> elements) {
      checkNotNull(elements, "elements == null");

      Iterator<? extends CodeElement> it = elements.iterator();
      if (it.hasNext()) {
        els.add(it.next());
        while (it.hasNext()) {
          els.add(CodeElements.COMMA);
          els.add(it.next());
        }
      }

      return self();
    }
    
    public final SELF withDot() {
      return withCodeElement(CodeElements.DOT);
    }

    public final SELF withIndentIfNecessary() {
      return withCodeElement(CodeElements.INDENT_IF_NECESSARY);
    }
    
    public final SELF withOpenBrace() {
      return withCodeElement(CodeElements.OPEN_BRACE);
    }
    
    public final SELF withOpenBracket() {
      return withCodeElement(CodeElements.OPEN_BRACKET);
    }

    public final SELF withSpaceOff() {
      return withCodeElement(CodeElements.SPACE_OFF);
    }
    
    public final SELF withTypeName(TypeName typeName) {
      return withCodeElement(CodeElements.ofTypeName(typeName));
    }
    
    public final SELF withWord(String word) {
      checkNotNull(word, "word == null");
      return withCodeElement(CodeElements.ofWord(word));
    }

    protected abstract SELF self();

    protected final ImmutableList<CodeElement> elements() {
      return els.toImmutableList();
    }

  }

}
