/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.element;

import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.TypeName;

final class CodeElements {

  static final CodeElement CLOSE_ANGLE = new CharCodeElement('>');

  static final CodeElement CLOSE_BRACE = new CharCodeElement('}');

  static final CodeElement CLOSE_BRACKET = new CharCodeElement(']');

  static final CodeElement CLOSE_PARENS = new CharCodeElement(')');

  static final CodeElement COLON = new CharCodeElement(':');

  static final CodeElement COMMA = new AbstractCodeElement() {
    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.spaceOff().writeWord(',');
    }
  };

  static final CodeElement DOT = new CharCodeElement('.');

  static final CodeElement INDENT_IF_NECESSARY = new AbstractCodeElement() {
    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.writePreIndentation();
    }
  };

  static final CodeElement OPEN_ANGLE = new CharCodeElement('<');

  static final CodeElement OPEN_BRACE = new CharCodeElement('{');

  static final CodeElement OPEN_BRACKET = new CharCodeElement('[');

  static final CodeElement OPEN_PARENS = new CharCodeElement('(');

  static final CodeElement QUESTION_MARK = new CharCodeElement('?');

  static final CodeElement SPACE_OFF = new AbstractCodeElement() {
    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.spaceOff();
    }
  };

  static final CodeElement SPACE_ON = new AbstractCodeElement() {
    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.spaceOn();
    }
  };
  
  private CodeElements() {}

  static CodeElement ofRaw(String string) {
    return new RawString(string);
  }

  static CodeElement ofStringLiteral(String s) {
    return new StringLiteral(s);
  }

  static CodeElement ofTypeName(TypeName typeName) {
    return new TypeNameCodeElement(typeName);
  }

  static CodeElement ofWord(char c) {
    return new CharWordCodeElement(c);
  }

  static CodeElement ofWord(String word) {
    return new SingleWordCodeElement(word);
  }
  
  private static class CharWordCodeElement extends AbstractCodeElement {
    private final char value;

    CharWordCodeElement(char value) {
      this.value = value;
    }
    
    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.writeWord(value);
    }
  }

  private static class RawString extends AbstractCodeElement {
    private final String value;

    RawString(String value) {
      this.value = value;
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.write(value);
    }
  }

  private static class StringLiteral extends AbstractCodeElement {
    private final String value;

    StringLiteral(String value) {
      this.value = value;
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.writeStringLiteral(value);
    }
  }

  private static class TypeNameCodeElement extends AbstractCodeElement {
    private final TypeName typeName;

    TypeNameCodeElement(TypeName typeName) {
      this.typeName = typeName;
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.writeTypeNameAsWord(typeName);
    }
  }

  private static class SingleWordCodeElement extends AbstractCodeElement {
    private final String value;

    SingleWordCodeElement(String value) {
      this.value = value;
    }

    @Override
    public final CodeWriter acceptCodeWriter(CodeWriter w) {
      return w.writeWord(value);
    }
  }

}
