/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.expression;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.CodeJava;
import br.com.objectos.code.java.element.CodeElement;
import br.com.objectos.code.java.element.NewLine;
import br.com.objectos.code.java.statement.VariableInitializer;
import br.com.objectos.comuns.collections.GrowableList;

public interface ExpressionCode extends CodeElement, LambdaBody, VariableInitializer {

  static Builder builder() {
    return new Builder();
  }

  static Builder builder(ExpressionCode startingExpression) {
    return builder().withExpression(startingExpression);
  }

  static class Builder {

    private final GrowableList<CodeElement> elements = GrowableList.newList();
    private boolean chain;

    private Builder() {}

    public final Expression build() {
      return new Expression(elements.toImmutableList());
    }

    public final Builder and(InclusiveOrExpression rhs) {
      return add(Operator.AND).add(rhs).setChain();
    }

    public final Builder id(String name) {
      return add(CodeJava.id(name)).setChain();
    }

    public final Builder invoke(String methodName, Argument... args) {
      checkNotNull(methodName, "methodName == null");
      checkNotNull(args, "args == null");
      return invoke0(methodName, Arguments.of(args));
    }

    public final Builder invoke(String methodName, Iterable<? extends Argument> args) {
      checkNotNull(methodName, "methodName == null");
      checkNotNull(args, "args == null");
      return invoke0(methodName, Arguments.of(args));
    }

    public final Builder nl() {
      return add(NewLine.nextLine());
    }

    final Builder withExpression(ExpressionCode expression) {
      return add(expression).setChain();
    }

    private Builder add(CodeElement element) {
      elements.add(element);
      return this;
    }

    private Builder invoke0(String methodName, Arguments args) {
      CodeElement invocation
          = chain
              ? ChainedMethodInvocation.ofUnchecked(methodName, args)
              : MethodInvocationImpl.invoke1(methodName, args);
      return add(invocation).setChain();
    }

    private Builder setChain() {
      chain = true;
      return this;
    }

  }

}