/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.statement;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.expression.ExpressionCode;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.statement.Block.Semicolon;

public class IfThenStatement extends AbstractStatement {

  private final ExpressionCode expression;
  private final Statement statement;

  private IfThenStatement(ExpressionCode expression, Statement statement) {
    this.expression = expression;
    this.statement = statement;
  }

  static Builder builder(ExpressionCode condition) {
    checkNotNull(condition, "condition == null");
    return new Builder(condition);
  }

  public final IfThenElseStatement _else(BlockElement... elements) {
    Block block = Block.of(elements);
    return new IfThenElseStatement(this, block);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w
        .writeControl("if", expression)
        .writeCodeElement(statement);
  }

  @Override
  public final void acceptSemicolon(Semicolon semicolon) {
    statement.acceptSemicolon(semicolon);
  }

  public static class Builder {

    private final ExpressionCode condition;

    private Builder(ExpressionCode condition) {
      this.condition = condition;
    }

    public final IfThenStatement then(BlockElement... elements) {
      return block(elements);
    }

    public final IfThenStatement block(BlockElement... elements) {
      return withBlock(Block.of(elements));
    }

    public final IfThenStatement withBlock(Block block) {
      checkNotNull(block, "block == null");
      return new IfThenStatement(condition, block);
    }

  }

}