/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.type;

import br.com.objectos.code.java.expression.CastExpression;
import br.com.objectos.code.java.expression.Expressions;
import br.com.objectos.code.java.expression.UnaryExpression;
import br.com.objectos.code.java.io.JavaFileImportSet;

public enum PrimitiveTypeName implements TypeName {

  BOOLEAN("boolean", Boolean.class),
  BYTE("byte", Byte.class),
  SHORT("short", Short.class),
  INT("int", Integer.class),
  LONG("long", Long.class),
  CHAR("char", Character.class),
  FLOAT("float", Float.class),
  DOUBLE("double", Double.class);

  private final String name;
  private final Class<?> wrapperClass;

  private PrimitiveTypeName(String name, Class<?> wrapperClass) {
    this.name = name;
    this.wrapperClass = wrapperClass;
  }

  public static PrimitiveTypeName _boolean() {
    return PrimitiveTypeName.BOOLEAN;
  }

  public static PrimitiveTypeName _byte() {
    return PrimitiveTypeName.BYTE;
  }

  public static PrimitiveTypeName _short() {
    return PrimitiveTypeName.SHORT;
  }

  public static PrimitiveTypeName _int() {
    return PrimitiveTypeName.INT;
  }

  public static PrimitiveTypeName _long() {
    return PrimitiveTypeName.LONG;
  }

  public static PrimitiveTypeName _char() {
    return PrimitiveTypeName.CHAR;
  }

  public static PrimitiveTypeName _float() {
    return PrimitiveTypeName.FLOAT;
  }

  public static PrimitiveTypeName _double() {
    return PrimitiveTypeName.DOUBLE;
  }

  @Override
  public final String acceptJavaFileImportSet(JavaFileImportSet set) {
    return name;
  }

  @Override
  public final <R, P> R acceptTypeNameVisitor(TypeNameVisitor<R, P> visitor, P p) {
    return visitor.visitPrimitiveType(this, p);
  }

  @Override
  public final TypeName arrayCreationTypeName() {
    return this;
  }
  
  public final CastExpression cast(UnaryExpression expression) {
    return Expressions.cast(this, expression);
  }

  public final ArrayTypeName toArrayTypeName() {
    return ArrayTypeName.ofUnchecked(this);
  }

  @Override
  public final String toString() {
    return name;
  }

  public final ClassName wrapperClassName() {
    return ClassName.of(wrapperClass);
  }

}