/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.type;

import java.util.List;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.PrimitiveType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.SimpleTypeVisitor8;

class TypeNameFactory extends SimpleTypeVisitor8<TypeName, Void> {

  private static final TypeNameFactory INSTANCE = new TypeNameFactory();

  private TypeNameFactory() {}

  static TypeName ofUnchecked(TypeMirror type) {
    return type.accept(INSTANCE, null);
  }

  @Override
  public final TypeName visitDeclared(DeclaredType t, Void p) {
    TypeElement typeElement = (TypeElement) t.asElement();
    ClassNameOrParameterizedTypeName typeName = ClassName.of(typeElement);

    List<? extends TypeMirror> typeArgumentList = t.getTypeArguments();
    for (TypeMirror typeArgument : typeArgumentList) {
      typeName = typeName.withTypeArgument(typeArgument.accept(this, p));
    }

    return typeName;
  }

  @Override
  public final TypeName visitPrimitive(PrimitiveType t, Void p) {
    switch (t.getKind()) {
      case BOOLEAN:
        return PrimitiveTypeName._boolean();
      case BYTE:
        return PrimitiveTypeName._byte();
      case SHORT:
        return PrimitiveTypeName._short();
      case INT:
        return PrimitiveTypeName._int();
      case LONG:
        return PrimitiveTypeName._long();
      case CHAR:
        return PrimitiveTypeName._char();
      case FLOAT:
        return PrimitiveTypeName._float();
      case DOUBLE:
        return PrimitiveTypeName._double();
      default:
        throw new AssertionError();
    }
  }

  @Override
  protected final TypeName defaultAction(TypeMirror e, Void p) {
    throw new UnsupportedOperationException("Implement me");
  }

}