/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.type;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.comuns.collections.ImmutableList;
import br.com.objectos.comuns.collections.StreamIterable;
import java.util.List;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.TypeParameterElement;
import javax.lang.model.type.TypeMirror;

public abstract class TypeParameterName extends AbstractCodeElement {

  TypeParameterName() {}

  public static TypeParameterName named(String name) {
    checkNotNull(name, "name == null");
    return new Unbounded(name);
  }

  public static TypeParameterName of(TypeParameterElement element) {
    checkNotNull(element, "element == null");
    TypeParameterName name = named(element.getSimpleName().toString());

    List<? extends TypeMirror> bounds = element.getBounds();
    for (TypeMirror bound : bounds) {
      TypeName typeName = TypeNameFactory.ofUnchecked(bound);
      if (!typeName.isJavaLangObject()) {
        name = name.addBound0(typeName);
      }
    }

    return name;
  }

  public static ImmutableList<TypeParameterName> immutableListOf(TypeElement typeElement) {
    return streamIterableOf(typeElement).toImmutableList();
  }

  public static StreamIterable<TypeParameterName> streamIterableOf(TypeElement typeElement) {
    checkNotNull(typeElement, "typeElement == null");
    return StreamIterable.adapt(typeElement.getTypeParameters()).map(TypeParameterName::of);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w.write(toString());
  }

  public final TypeParameterName addBound(Class<?> type) {
    return addBound0(ClassName.of(type));
  }

  @Override
  public final boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (!(obj instanceof TypeParameterName)) {
      return false;
    }
    TypeParameterName that = (TypeParameterName) obj;
    return toString().equals(that.toString());
  }

  @Override
  public final int hashCode() {
    return toString().hashCode();
  }

  @Override
  public abstract String toString();

  public abstract TypeVariableName toTypeVariableName();

  abstract TypeParameterName addBound0(TypeName typeName);

  private static class Unbounded extends TypeParameterName {

    private final String name;

    Unbounded(String name) {
      this.name = name;
    }

    @Override
    public final String toString() {
      return name;
    }

    @Override
    public final TypeVariableName toTypeVariableName() {
      return TypeVariableName.named(name);
    }

    @Override
    final TypeParameterName addBound0(TypeName typeName) {
      return Bounded.first(this, typeName);
    }

  }

  private static class Bounded extends TypeParameterName {

    private final TypeParameterName previous;
    private final String separator;
    private final TypeName bound;

    Bounded(TypeParameterName previous, String separator, TypeName bound) {
      this.previous = previous;
      this.separator = separator;
      this.bound = bound;
    }

    static Bounded first(Unbounded unbounded, TypeName bound) {
      return new Bounded(unbounded, " extends ", bound);
    }

    @Override
    public final String toString() {
      return previous + separator + bound;
    }

    @Override
    public final TypeVariableName toTypeVariableName() {
      return previous.toTypeVariableName();
    }

    @Override
    final TypeParameterName addBound0(TypeName typeName) {
      return new Bounded(this, " & ", typeName);
    }

  }

}