/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.io.BodyFormatter;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.TypeParameterName;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;

public final class InterfaceCode extends TypeCode {

  private final InterfaceModifier modifier;
  private final String simpleName;
  private final ImmutableList<TypeParameterName> typeParameters = ImmutableList.empty();
  private final ImmutableList<InterfaceBodyElement> bodyElements;

  private InterfaceCode(Builder builder) {
    modifier = builder.modifier();
    simpleName = builder.simpleName;
    bodyElements = builder.bodyElements();
  }

  public static Builder _interface(String simpleName) {
    return builder()._interface(simpleName);
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w
        .pushSimpleName(simpleName)
        .writeCodeElement(modifier)
        .writeWord("interface")
        .writeSimpleNameWith(typeParameters)
        .writeBlock(bodyElements)
        .popSimpleName();
  }

  @Override
  public final String simpleName() {
    return simpleName;
  }

  public static final class Builder {

    private final InterfaceModifier.Builder modifier = InterfaceModifier.builder();
    private String simpleName = "Unnamed";
    private final GrowableList<InterfaceBodyElement> bodyElements = GrowableList.newList();

    private BodyFormatter formatter = BodyFormatter.defaultFormatter();

    private Builder() {}

    public final Builder _interface(String simpleName) {
      checkNotNull(simpleName, "simpleName == null");
      this.simpleName = simpleName;
      return this;
    }

    public final Builder body(InterfaceBodyElement... elements) {
      bodyElements.fluentAdd(elements);
      return this;
    }

    public final Builder body(Iterable<InterfaceBodyElement> elements) {
      return withBodyElements(elements);
    }

    public final InterfaceCode build() {
      return new InterfaceCode(this);
    }

    public final InterfaceCode buildWith(BodyFormatter formatter) {
      this.formatter = checkNotNull(formatter, "formatter == null");
      return build();
    }

    public final Builder withBodyElements(Iterable<InterfaceBodyElement> elements) {
      bodyElements.addAll(elements);
      return this;
    }

    public final Builder withModifier(InterfaceModifier modifier) {
      checkNotNull(modifier, "modifier == null");
      this.modifier.withModifier(modifier);
      return this;
    }

    final ImmutableList<InterfaceBodyElement> bodyElements() {
      return formatter.format(bodyElements, InterfaceBodyElement.class);
    }

    final InterfaceModifier modifier() {
      return modifier.build();
    }

  }

}