/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.type.EnclosingElement;
import javax.lang.model.element.PackageElement;

public class PackageName implements EnclosingElement {

  private static final PackageName UNNAMED = new PackageName("");

  private final String packageName;

  private PackageName(String packageName) {
    this.packageName = packageName;
  }

  public static PackageName named(String name) {
    checkNotNull(name, "name == null");
    return new PackageName(name);
  }

  public static PackageName of(Class<?> type) {
    checkNotNull(type, "type == null");
    return of(type.getPackage());
  }

  public static PackageName of(Package pkg) {
    checkNotNull(pkg, "pkg == null");
    return new PackageName(pkg.getName());
  }

  public static PackageName of(PackageElement element) {
    checkNotNull(element, "element == null");
    return new PackageName(element.getQualifiedName().toString());
  }

  public static PackageName unnamed() {
    return UNNAMED;
  }

  @Override
  public final boolean equals(Object obj) {
    if (!(obj instanceof PackageName)) {
      return false;
    }
    PackageName that = (PackageName) obj;
    return packageName.equals(that.packageName);
  }

  @Override
  public final int hashCode() {
    return packageName.hashCode();
  }

  public final boolean is(String name) {
    return packageName.equals(name);
  }

  public final boolean isUnnamed() {
    return is("");
  }

  @Override
  public final PackageName packageName() {
    return this;
  }

  @Override
  public final String toString() {
    return packageName;
  }

}