/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.expression;

import br.com.objectos.code.java.element.AbstractDefaultCodeElement;
import br.com.objectos.code.java.element.CodeElement;
import br.com.objectos.code.java.element.ImmutableCodeElement;
import br.com.objectos.code.java.type.ArrayTypeName;
import br.com.objectos.comuns.collections.GrowableList;
import java.util.Arrays;
import java.util.Iterator;

final class ArrayCreationExpressionImpl
    extends AbstractDefaultCodeElement
    implements ArrayCreationExpression {

  private ArrayCreationExpressionImpl(ImmutableCodeElement delegate) {
    super(delegate);
  }

  static ArrayCreationExpression _new0(ArrayTypeName type, ExpressionCode... dims) {
    return _new0(type, Arrays.asList(dims));
  }

  static ArrayCreationExpression _new0(
      ArrayTypeName type, Iterable<? extends ExpressionCode> dims) {
    return build0(
        ImmutableCodeElement.builder()
            .withWord("new")
            .withTypeName(type.arrayCreationTypeName())
            .withCodeElements(dims(type, dims))
            .build()
    );
  }

  private static ArrayCreationExpression build0(ImmutableCodeElement delegate) {
    return new ArrayCreationExpressionImpl(delegate);
  }

  private static CodeElement dim(ExpressionCode dim) {
    return ImmutableCodeElement.builder()
        .withSpaceOff()
        .withOpenBracket()
        .withSpaceOff()
        .withCodeElement(dim)
        .withSpaceOff()
        .withCloseBracket()
        .build();
  }

  private static Iterable<CodeElement> dims(
      ArrayTypeName type, Iterable<? extends ExpressionCode> dims) {
    GrowableList<CodeElement> dimList = GrowableList.newList();
    Iterator<? extends ExpressionCode> it = dims.iterator();

    int dimension = type.dimension();
    for (int i = 0; i < dimension; i++) {
      ExpressionCode dim;
      if (it.hasNext()) {
        dim = it.next();
      } else {
        dim = EmptyExpression.empty();
      }
      dimList.add(dim(dim));
    }

    if (it.hasNext()) {
      throw new IllegalArgumentException("Too many dim expressions: array dimension=" + dimension);
    }

    return dimList;
  }

  @Override
  public final FieldAccess id(Identifier id) {
    return Expressions.fieldAccess(this, id);
  }

  @Override
  public final FieldAccess id(String id) {
    return Expressions.fieldAccess(this, id);
  }

  @Override
  protected final ArrayReferenceExpression selfArrayReferenceExpression() {
    throw newUoe(ArrayCreationExpression.class);
  }

  @Override
  protected final ConditionalAndExpression selfConditionalAndExpression() {
    return this;
  }

  @Override
  protected final Callee selfCallee() {
    throw newUoe(ArrayCreationExpression.class);
  }

  @Override
  protected final LeftHandSide selfLeftHandSide() {
    throw newUoe(ArrayCreationExpression.class);
  }

  @Override
  protected final MethodReferenceReferenceExpression selfMethodReferenceReferenceExpression() {
    return this;
  }

  @Override
  protected final MultiplicativeExpression selfMultiplicativeExpression() {
    return this;
  }

  @Override
  protected final PostfixExpression selfPostfixExpression() {
    return this;
  }

  @Override
  protected final RelationalExpression selfRelationalExpression() {
    return this;
  }

}
