/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.expression;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.declaration.ParameterCode;
import br.com.objectos.code.java.statement.VariableInitializer;
import br.com.objectos.code.java.type.ArrayTypeName;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.PrimitiveTypeName;
import br.com.objectos.code.java.type.ReferenceTypeName;
import br.com.objectos.code.java.type.TypeName;

public class Expressions {

  private Expressions() {}

  // AdditiveExpression

  public static AdditiveExpression add(AdditiveExpression lhs, MultiplicativeExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return AdditiveExpressionImpl.of0(AdditiveOperator.ADDITION, lhs, rhs);
  }

  public static AdditiveExpression subtract(AdditiveExpression lhs, MultiplicativeExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return AdditiveExpressionImpl.of0(AdditiveOperator.SUBTRACTION, lhs, rhs);
  }

  // AndExpression

  public static AndExpression bitwiseAnd(AndExpression lhs, EqualityExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return AndExpressionImpl.bitwiseAnd0(lhs, rhs);
  }
  
  // ArrayAccess

  public static ArrayAccess aget(ArrayReferenceExpression ref, ExpressionCode e0) {
    return arrayAccess(ref, e0);
  }

  public static ArrayAccess aget(ArrayReferenceExpression ref,
      ExpressionCode e0, ExpressionCode e1) {
    return arrayAccess(ref, e0, e1);
  }

  public static ArrayAccess aget(ArrayReferenceExpression ref,
      ExpressionCode e0, ExpressionCode e1, ExpressionCode e2) {
    return arrayAccess(ref, e0, e1, e2);
  }

  public static ArrayAccess aget(ArrayReferenceExpression ref,
      ExpressionCode e0, ExpressionCode e1, ExpressionCode e2, ExpressionCode e3) {
    return arrayAccess(ref, e0, e1, e2, e3);
  }

  public static ArrayAccess aget(
      ArrayReferenceExpression ref, Iterable<? extends ExpressionCode> expressions) {
    return arrayAccess(ref, expressions);
  }

  public static ArrayAccess arrayAccess(ArrayReferenceExpression ref, ExpressionCode e0) {
    checkNotNull(ref, "ref == null");
    checkNotNull(e0, "e0 == null");
    return ArrayAccessImpl.arrayAccess0(ref, e0);
  }

  public static ArrayAccess arrayAccess(
      ArrayReferenceExpression ref,
      ExpressionCode e0, ExpressionCode e1) {
    checkNotNull(ref, "ref == null");
    checkNotNull(e0, "e0 == null");
    checkNotNull(e1, "e1 == null");
    return ArrayAccessImpl.arrayAccess0(ref, e0, e1);
  }

  public static ArrayAccess arrayAccess(
      ArrayReferenceExpression ref,
      ExpressionCode e0, ExpressionCode e1, ExpressionCode e2) {
    checkNotNull(ref, "ref == null");
    checkNotNull(e0, "e0 == null");
    checkNotNull(e1, "e1 == null");
    checkNotNull(e2, "e2 == null");
    return ArrayAccessImpl.arrayAccess0(ref, e0, e1, e2);
  }

  public static ArrayAccess arrayAccess(
      ArrayReferenceExpression ref,
      ExpressionCode e0, ExpressionCode e1, ExpressionCode e2, ExpressionCode e3) {
    checkNotNull(ref, "ref == null");
    checkNotNull(e0, "e0 == null");
    checkNotNull(e1, "e1 == null");
    checkNotNull(e2, "e2 == null");
    checkNotNull(e3, "e3 == null");
    return ArrayAccessImpl.arrayAccess0(ref, e0, e1, e2, e3);
  }

  public static ArrayAccess arrayAccess(
      ArrayReferenceExpression ref,
      Iterable<? extends ExpressionCode> expressions) {
    checkNotNull(ref, "ref == null");
    checkNotNull(expressions, "expressions == null");
    return ArrayAccessImpl.arrayAccess0(ref, expressions);
  }

  // ArrayCreationExpression

  public static ArrayCreationExpression _new(
      ArrayTypeName type, ExpressionCode dim0) {
    checkNotNull(type, "type == null");
    checkNotNull(dim0, "dim0 == null");
    return ArrayCreationExpressionImpl._new0(type, dim0);
  }

  public static ArrayCreationExpression _new(
      ArrayTypeName type, ExpressionCode dim0, ExpressionCode dim1) {
    checkNotNull(type, "type == null");
    checkNotNull(dim0, "dim0 == null");
    checkNotNull(dim1, "dim1 == null");
    return ArrayCreationExpressionImpl._new0(type, dim0, dim1);
  }

  public static ArrayCreationExpression _new(
      ArrayTypeName type, ExpressionCode dim0, ExpressionCode dim1, ExpressionCode dim2) {
    checkNotNull(type, "type == null");
    checkNotNull(dim0, "dim0 == null");
    checkNotNull(dim1, "dim1 == null");
    checkNotNull(dim2, "dim2 == null");
    return ArrayCreationExpressionImpl._new0(type, dim0, dim1, dim2);
  }

  public static ArrayCreationExpression _new(
      ArrayTypeName type,
      ExpressionCode dim0, ExpressionCode dim1, ExpressionCode dim2, ExpressionCode dim3) {
    checkNotNull(type, "type == null");
    checkNotNull(dim0, "dim0 == null");
    checkNotNull(dim1, "dim1 == null");
    checkNotNull(dim2, "dim2 == null");
    checkNotNull(dim3, "dim3 == null");
    return ArrayCreationExpressionImpl._new0(type, dim0, dim1, dim2, dim3);
  }

  public static ArrayCreationExpression _new(
      ArrayTypeName type, Iterable<? extends ExpressionCode> dims) {
    checkNotNull(type, "type == null");
    checkNotNull(dims, "dims == null");
    return ArrayCreationExpressionImpl._new0(type, dims);
  }

  // ArrayInitializer

  public static ArrayInitializer a() {
    return ArrayInitializerImpl.EMPTY;
  }

  public static ArrayInitializer a(
      VariableInitializer v0) {
    checkNotNull(v0, "v0 == null");
    return ArrayInitializerImpl.a0(v0);
  }

  public static ArrayInitializer a(
      VariableInitializer v0,
      VariableInitializer v1) {
    checkNotNull(v0, "v0 == null");
    checkNotNull(v1, "v1 == null");
    return ArrayInitializerImpl.a0(v0, v1);
  }

  public static ArrayInitializer a(
      VariableInitializer v0,
      VariableInitializer v1,
      VariableInitializer v2) {
    checkNotNull(v0, "v0 == null");
    checkNotNull(v1, "v1 == null");
    checkNotNull(v2, "v2 == null");
    return ArrayInitializerImpl.a0(v0, v1, v2);
  }

  public static ArrayInitializer a(
      VariableInitializer v0,
      VariableInitializer v1,
      VariableInitializer v2,
      VariableInitializer v3) {
    checkNotNull(v0, "v0 == null");
    checkNotNull(v1, "v1 == null");
    checkNotNull(v2, "v2 == null");
    checkNotNull(v3, "v3 == null");
    return ArrayInitializerImpl.a0(v0, v1, v2, v3);
  }

  public static ArrayInitializer a(
      VariableInitializer v0,
      VariableInitializer v1,
      VariableInitializer v2,
      VariableInitializer v3,
      VariableInitializer v4) {
    checkNotNull(v0, "v0 == null");
    checkNotNull(v1, "v1 == null");
    checkNotNull(v2, "v2 == null");
    checkNotNull(v3, "v3 == null");
    checkNotNull(v4, "v4 == null");
    return ArrayInitializerImpl.a0(v0, v1, v2, v3, v4);
  }

  public static ArrayInitializer a(
      VariableInitializer v0,
      VariableInitializer v1,
      VariableInitializer v2,
      VariableInitializer v3,
      VariableInitializer v4,
      VariableInitializer v5) {
    checkNotNull(v0, "v0 == null");
    checkNotNull(v1, "v1 == null");
    checkNotNull(v2, "v2 == null");
    checkNotNull(v3, "v3 == null");
    checkNotNull(v4, "v4 == null");
    checkNotNull(v5, "v5 == null");
    return ArrayInitializerImpl.a0(v0, v1, v2, v3, v4, v5);
  }

  public static ArrayInitializer a(
      VariableInitializer v0,
      VariableInitializer v1,
      VariableInitializer v2,
      VariableInitializer v3,
      VariableInitializer v4,
      VariableInitializer v5,
      VariableInitializer v6) {
    checkNotNull(v0, "v0 == null");
    checkNotNull(v1, "v1 == null");
    checkNotNull(v2, "v2 == null");
    checkNotNull(v3, "v3 == null");
    checkNotNull(v4, "v4 == null");
    checkNotNull(v5, "v5 == null");
    checkNotNull(v6, "v6 == null");
    return ArrayInitializerImpl.a0(v0, v1, v2, v3, v4, v5, v6);
  }

  public static ArrayInitializer a(VariableInitializer... elements) {
    checkNotNull(elements, "elements == null");
    switch (elements.length) {
      case 0:
        return a();
      case 1:
        return a(
            elements[0]
        );
      case 2:
        return a(
            elements[0],
            elements[1]
        );
      case 3:
        return a(
            elements[0],
            elements[1],
            elements[2]
        );
      case 4:
        return a(
            elements[0],
            elements[1],
            elements[2],
            elements[3]
        );
      case 5:
        return a(
            elements[0],
            elements[1],
            elements[2],
            elements[3],
            elements[4]
        );
      case 6:
        return a(
            elements[0],
            elements[1],
            elements[2],
            elements[3],
            elements[4],
            elements[5]
        );
      case 7:
        return a(
            elements[0],
            elements[1],
            elements[2],
            elements[3],
            elements[4],
            elements[5],
            elements[6]
        );
      default:
        for (int i = 0; i < elements.length; i++) {
          checkNotNull(elements[i], "v" + i + " == null");
        }
        return ArrayInitializerImpl.a0(elements);
    }
  }

  // Assignment

  public static Assignment assign(LeftHandSide lhs, ExpressionCode rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return AssignmentImpl.assign0(AssignmentOperator.SIMPLE, lhs, rhs);
  }

  public static Assignment assign(
      AssignmentOperator operator, LeftHandSide lhs, ExpressionCode rhs) {
    checkNotNull(operator, "operator == null");
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return AssignmentImpl.assign0(operator, lhs, rhs);
  }

  // CastExpression

  public static CastExpression cast(
      ReferenceTypeName type, UnaryExpressionNotPlusMinus expression) {
    checkNotNull(type, "type == null");
    checkNotNull(expression, "expression == null");
    return CastExpressionImpl.cast0(type, expression);
  }

  public static CastExpression cast(
      PrimitiveTypeName type, UnaryExpression expression) {
    checkNotNull(type, "type == null");
    checkNotNull(expression, "expression == null");
    return CastExpressionImpl.cast0(type, expression);
  }

  // ClassInstanceCreationExpression

  public static ClassInstanceCreationExpression _new(
      ClassName className) {
    checkNotNull(className, "className == null");
    return ClassInstanceCreationExpressionImpl.new0(className);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className,
      Argument a1) {
    checkNotNull(className, "className == null");
    checkNotNull(a1, "a1 == null");
    return ClassInstanceCreationExpressionImpl.new0(className, a1);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className,
      Argument a1, Argument a2) {
    checkNotNull(className, "className == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    return ClassInstanceCreationExpressionImpl.new0(className, a1, a2);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className,
      Argument a1, Argument a2, Argument a3) {
    checkNotNull(className, "className == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    return ClassInstanceCreationExpressionImpl.new0(className, a1, a2, a3);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className,
      Argument a1, Argument a2, Argument a3, Argument a4) {
    checkNotNull(className, "className == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    checkNotNull(a4, "a4 == null");
    return ClassInstanceCreationExpressionImpl.new0(className, a1, a2, a3, a4);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className,
      Iterable<? extends Argument> args) {
    checkNotNull(className, "className == null");
    checkNotNull(args, "args == null");
    return ClassInstanceCreationExpressionImpl.new0(className, args);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className, TypeWitness witness) {
    checkNotNull(className, "className == null");
    checkNotNull(witness, "witness == null");
    return ClassInstanceCreationExpressionImpl.new0(className, witness);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className, TypeWitness witness,
      Argument a1) {
    checkNotNull(className, "className == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(a1, "a1 == null");
    return ClassInstanceCreationExpressionImpl.new0(className, witness, a1);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className, TypeWitness witness,
      Argument a1, Argument a2) {
    checkNotNull(className, "className == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    return ClassInstanceCreationExpressionImpl.new0(className, witness, a1, a2);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className, TypeWitness witness,
      Argument a1, Argument a2, Argument a3) {
    checkNotNull(className, "className == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    return ClassInstanceCreationExpressionImpl.new0(className, witness, a1, a2, a3);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className, TypeWitness witness,
      Argument a1, Argument a2, Argument a3, Argument a4) {
    checkNotNull(className, "className == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    checkNotNull(a4, "a4 == null");
    return ClassInstanceCreationExpressionImpl.new0(className, witness, a1, a2, a3, a4);
  }

  public static ClassInstanceCreationExpression _new(
      ClassName className, TypeWitness witness,
      Iterable<? extends Argument> args) {
    checkNotNull(className, "className == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(args, "args == null");
    return ClassInstanceCreationExpressionImpl.new0(className, witness, args);
  }

  // ConditionalAndExpression

  public static ConditionalAndExpression and(
      ConditionalAndExpression lhs, InclusiveOrExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return ConditionalAndExpressionImpl.and0(lhs, rhs);
  }

  // ConditionalExpression

  public static ConditionalExpression ternary(
      ConditionalOrExpression condition,
      ExpressionCode trueExpression, ConditionalExpression falseExpression) {
    checkNotNull(condition, "condition == null");
    checkNotNull(trueExpression, "trueExpression == null");
    checkNotNull(falseExpression, "falseExpression == null");
    return ConditionalExpressionImpl.ternary0(condition, trueExpression, falseExpression);
  }

  public static ConditionalExpression ternary(
      ConditionalOrExpression condition,
      ExpressionCode trueExpression, LambdaExpression falseExpression) {
    checkNotNull(condition, "condition == null");
    checkNotNull(trueExpression, "trueExpression == null");
    checkNotNull(falseExpression, "falseExpression == null");
    return ConditionalExpressionImpl.ternary0(condition, trueExpression, falseExpression);
  }
  
  // ConditionalOrExpression

  public static ConditionalOrExpression or(
      ConditionalOrExpression lhs, ConditionalAndExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return ConditionalOrExpressionImpl.or0(lhs, rhs);
  }

  // ExclusiveOrExpression

  public static ExclusiveOrExpression bitwiseXor(ExclusiveOrExpression lhs, AndExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return ExclusiveOrExpressionImpl.bitwiseXor0(lhs, rhs);
  }
  
  // EqualityExpression

  public static EqualityExpression eq(EqualityExpression lhs, RelationalExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return EqualityExpressionImpl.eq0(lhs, rhs);
  }

  public static EqualityExpression ne(EqualityExpression lhs, RelationalExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return EqualityExpressionImpl.ne0(lhs, rhs);
  }

  // ExpresssionName

  public static ExpressionName expressionName(Class<?> type, String id) {
    checkNotNull(type, "type == null");
    checkNotNull(id, "id == null");
    return ExpressionNameImpl.expressionName0(ClassName.of(type), id(id));
  }

  public static ExpressionName expressionName(ClassName className, Identifier id) {
    checkNotNull(className, "className == null");
    checkNotNull(id, "id == null");
    return ExpressionNameImpl.expressionName0(className, id);
  }

  public static ExpressionName expressionName(ClassName className, String id) {
    checkNotNull(className, "className == null");
    checkNotNull(id, "id == null");
    return ExpressionNameImpl.expressionName0(className, id(id));
  }

  public static ExpressionName expressionName(ExpressionName name, String id) {
    checkNotNull(name, "name == null");
    checkNotNull(id, "id == null");
    return ExpressionNameImpl.expressionName0(name, id(id));
  }

  public static ExpressionName expressionName(ExpressionName name, Identifier id) {
    checkNotNull(name, "name == null");
    checkNotNull(id, "id == null");
    return ExpressionNameImpl.expressionName0(name, id);
  }

  // FieldAccess

  public static FieldAccess fieldAccess(FieldAccessReferenceExpression ref, String id) {
    checkNotNull(ref, "ref == null");
    checkNotNull(id, "id == null");
    return FieldAccessImpl.fieldAccess0(ref, id(id));
  }

  public static FieldAccess fieldAccess(FieldAccessReferenceExpression ref, Identifier id) {
    checkNotNull(ref, "ref == null");
    checkNotNull(id, "id == null");
    return FieldAccessImpl.fieldAccess0(ref, id);
  }

  // Identifier

  public static Identifier id(String name) {
    checkNotNull(name, "name == null");
    return IdentifierImpl.id0(name);
  }

  // InclusiveOrExpression

  public static InclusiveOrExpression bitwiseOr(
      InclusiveOrExpression lhs, ExclusiveOrExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return InclusiveOrExpressionImpl.bitwiseOr0(lhs, rhs);
  }
  
  // LambdaExpression

  public static LambdaExpression lambda(
      LambdaBody body) {
    checkNotNull(body, "body == null");
    return LambdaExpressionImpl.lambda0(body);
  }

  public static LambdaExpression lambda(
      Identifier p1,
      LambdaBody body) {
    checkNotNull(p1, "p1 == null");
    checkNotNull(body, "body == null");
    return LambdaExpressionImpl.lambda0(body, p1);
  }

  public static LambdaExpression lambda(
      ParameterCode p1,
      LambdaBody body) {
    checkNotNull(p1, "p1 == null");
    checkNotNull(body, "body == null");
    return LambdaExpressionImpl.lambda0(body, p1);
  }

  public static LambdaExpression lambda(
      LambdaParameter p1, LambdaParameter p2,
      LambdaBody body) {
    checkNotNull(p1, "p1 == null");
    checkNotNull(p2, "p2 == null");
    checkNotNull(body, "body == null");
    return LambdaExpressionImpl.lambda0(body, p1, p2);
  }

  public static LambdaExpression lambda(
      LambdaParameter p1, LambdaParameter p2, LambdaParameter p3,
      LambdaBody body) {
    checkNotNull(p1, "p1 == null");
    checkNotNull(p2, "p2 == null");
    checkNotNull(p3, "p3 == null");
    checkNotNull(body, "body == null");
    return LambdaExpressionImpl.lambda0(body, p1, p2, p3);
  }

  public static LambdaExpression lambda(
      LambdaParameter p1, LambdaParameter p2, LambdaParameter p3, LambdaParameter p4,
      LambdaBody body) {
    checkNotNull(p1, "p1 == null");
    checkNotNull(p2, "p2 == null");
    checkNotNull(p3, "p3 == null");
    checkNotNull(p4, "p4 == null");
    checkNotNull(body, "body == null");
    return LambdaExpressionImpl.lambda0(body, p1, p2, p3, p4);
  }

  public static LambdaExpression lambda(
      Iterable<? extends LambdaParameter> params,
      LambdaBody body) {
    checkNotNull(params, "params == null");
    checkNotNull(body, "body == null");
    return LambdaExpressionImpl.lambda0(body, params);
  }

  // Literal

  public static Literal l(boolean value) {
    return LiteralImpl.l0(value);
  }

  public static Literal l(char value) {
    return LiteralImpl.l0(value);
  }

  public static Literal l(double value) {
    return LiteralImpl.l0(value);
  }

  public static Literal l(int value) {
    return LiteralImpl.l0(value);
  }

  public static Literal l(float value) {
    return LiteralImpl.l0(value);
  }

  public static Literal l(long value) {
    return LiteralImpl.l0(value);
  }

  public static Literal l(Class<?> type) {
    checkNotNull(type, "type == null");
    return LiteralImpl.l0(type);
  }

  public static Literal l(ClassName className) {
    checkNotNull(className, "className == null");
    return LiteralImpl.l0(className);
  }

  public static Literal l(String s) {
    checkNotNull(s, "s == null");
    return LiteralImpl.l0(s);
  }

  // MethodReference

  public static MethodReference ref(
      MethodReferenceReferenceExpression expression, String methodName) {
    checkNotNull(expression, "expression == null");
    checkNotNull(methodName, "methodName == null");
    return MethodReferenceImpl.ref0(expression, methodName);
  }

  public static MethodReference ref(
      MethodReferenceReferenceExpression expression, TypeWitness witness, String methodName) {
    checkNotNull(expression, "expression == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(methodName, "methodName == null");
    return MethodReferenceImpl.ref0(expression, witness, methodName);
  }

  public static MethodReference ref(
      ReferenceTypeName typeName, String methodName) {
    checkNotNull(typeName, "typeName == null");
    checkNotNull(methodName, "methodName == null");
    return MethodReferenceImpl.ref0(typeName, methodName);
  }

  public static MethodReference ref(
      ReferenceTypeName typeName, TypeWitness witness, String methodName) {
    checkNotNull(typeName, "typeName == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(methodName, "methodName == null");
    return MethodReferenceImpl.ref0(typeName, witness, methodName);
  }

  // MethodInvocation

  public static MethodInvocation invoke(
      Callee callee, String methodName) {
    checkNotNull(callee, "callee == null");
    checkNotNull(methodName, "methodName == null");
    return MethodInvocationImpl.invoke0(callee, methodName);
  }

  public static MethodInvocation invoke(
      Callee callee, String methodName, Argument a1) {
    checkNotNull(callee, "callee == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    return MethodInvocationImpl.invoke0(callee, methodName, a1);
  }

  public static MethodInvocation invoke(
      Callee callee, String methodName, Argument a1, Argument a2) {
    checkNotNull(callee, "callee == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    return MethodInvocationImpl.invoke0(callee, methodName, a1, a2);
  }

  public static MethodInvocation invoke(
      Callee callee, String methodName, Argument a1, Argument a2, Argument a3) {
    checkNotNull(callee, "callee == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    return MethodInvocationImpl.invoke0(callee, methodName, a1, a2, a3);
  }

  public static MethodInvocation invoke(
      Callee callee, String methodName, Argument a1, Argument a2, Argument a3, Argument a4) {
    checkNotNull(callee, "callee == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    checkNotNull(a4, "a4 == null");
    return MethodInvocationImpl.invoke0(callee, methodName, a1, a2, a3, a4);
  }

  public static MethodInvocation invoke(
      Callee callee, String methodName, Iterable<? extends Argument> args) {
    checkNotNull(callee, "callee == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(args, "args == null");
    return MethodInvocationImpl.invoke0(callee, methodName, args);
  }

  public static MethodInvocation invoke(
      Callee callee, TypeWitness witness, String methodName) {
    checkNotNull(callee, "callee == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(methodName, "methodName == null");
    return MethodInvocationImpl.invoke0(callee, witness, methodName);
  }

  public static MethodInvocation invoke(
      Callee callee, TypeWitness witness, String methodName,
      Argument a1) {
    checkNotNull(callee, "callee == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    return MethodInvocationImpl.invoke0(callee, witness, methodName, a1);
  }

  public static MethodInvocation invoke(
      Callee callee, TypeWitness witness, String methodName,
      Argument a1, Argument a2) {
    checkNotNull(callee, "callee == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    return MethodInvocationImpl.invoke0(callee, witness, methodName, a1, a2);
  }

  public static MethodInvocation invoke(
      Callee callee, TypeWitness witness, String methodName,
      Argument a1, Argument a2, Argument a3) {
    checkNotNull(callee, "callee == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    return MethodInvocationImpl.invoke0(callee, witness, methodName, a1, a2, a3);
  }

  public static MethodInvocation invoke(
      Callee callee, TypeWitness witness, String methodName,
      Argument a1, Argument a2, Argument a3, Argument a4) {
    checkNotNull(callee, "callee == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    checkNotNull(a4, "a4 == null");
    return MethodInvocationImpl.invoke0(callee, witness, methodName, a1, a2, a3, a4);
  }

  public static MethodInvocation invoke(
      Callee callee, TypeWitness witness, String methodName, Iterable<? extends Argument> args) {
    checkNotNull(callee, "callee == null");
    checkNotNull(witness, "witness == null");
    checkNotNull(methodName, "methodName == null");
    checkNotNull(args, "args == null");
    return MethodInvocationImpl.invoke0(callee, witness, methodName, args);
  }

  public static MethodInvocation invoke(
      String methodName) {
    checkNotNull(methodName, "methodName == null");
    return MethodInvocationImpl.invoke0(methodName);
  }

  public static MethodInvocation invoke(
      String methodName,
      Argument a1) {
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    return MethodInvocationImpl.invoke0(methodName, a1);
  }

  public static MethodInvocation invoke(
      String methodName,
      Argument a1, Argument a2) {
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    return MethodInvocationImpl.invoke0(methodName, a1, a2);
  }

  public static MethodInvocation invoke(
      String methodName,
      Argument a1, Argument a2, Argument a3) {
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    return MethodInvocationImpl.invoke0(methodName, a1, a2, a3);
  }

  public static MethodInvocation invoke(
      String methodName,
      Argument a1, Argument a2, Argument a3, Argument a4) {
    checkNotNull(methodName, "methodName == null");
    checkNotNull(a1, "a1 == null");
    checkNotNull(a2, "a2 == null");
    checkNotNull(a3, "a3 == null");
    checkNotNull(a4, "a4 == null");
    return MethodInvocationImpl.invoke0(methodName, a1, a2, a3, a4);
  }

  public static MethodInvocation invoke(
      String methodName, Iterable<? extends Argument> args) {
    checkNotNull(methodName, "methodName == null");
    checkNotNull(args, "args == null");
    return MethodInvocationImpl.invoke0(methodName, args);
  }

  // MultiplicativeExpression

  public static MultiplicativeExpression divide(
      MultiplicativeExpression lhs, UnaryExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return MultiplicativeExpressionImpl.of0(MultiplicativeOperator.DIVISION, lhs, rhs);
  }

  public static MultiplicativeExpression multiply(
      MultiplicativeExpression lhs, UnaryExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return MultiplicativeExpressionImpl.of0(MultiplicativeOperator.MULTIPLICATION, lhs, rhs);
  }

  public static MultiplicativeExpression remainder(
      MultiplicativeExpression lhs, UnaryExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return MultiplicativeExpressionImpl.of0(MultiplicativeOperator.REMAINDER, lhs, rhs);
  }

  // ParenthesizedExpression

  public static ParenthesizedExpression parens(ExpressionCode expression) {
    checkNotNull(expression, "expression == null");
    return ParenthesizedExpressionImpl.parens0(expression);
  }

  // PostDecrementExpression

  public static PostDecrementExpression postDec(PostfixExpression expression) {
    checkNotNull(expression, "expression == null");
    return PostDecrementExpressionImpl.postDec0(expression);
  }

  // PostIncrementExpression

  public static PostIncrementExpression postInc(PostfixExpression expression) {
    checkNotNull(expression, "expression == null");
    return PostIncrementExpressionImpl.postInc0(expression);
  }

  // PreDecrementExpression

  public static PreDecrementExpression preDec(UnaryExpression expression) {
    checkNotNull(expression, "expression == null");
    return PreDecrementExpressionImpl.preDec0(expression);
  }

  // PreIncrementExpression

  public static PreIncrementExpression preInc(UnaryExpression expression) {
    checkNotNull(expression, "expression == null");
    return PreIncrementExpressionImpl.preInc0(expression);
  }

  // RelationalExpression

  public static RelationalExpression lt(
      RelationalExpression lhs, ShiftExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return RelationalExpressionImpl.of0(RelationalOperator.LT, lhs, rhs);
  }

  public static RelationalExpression gt(
      RelationalExpression lhs, ShiftExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return RelationalExpressionImpl.of0(RelationalOperator.GT, lhs, rhs);
  }

  public static RelationalExpression le(
      RelationalExpression lhs, ShiftExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return RelationalExpressionImpl.of0(RelationalOperator.LE, lhs, rhs);
  }

  public static RelationalExpression ge(
      RelationalExpression lhs, ShiftExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return RelationalExpressionImpl.of0(RelationalOperator.GE, lhs, rhs);
  }

  public static RelationalExpression instanceOf(
      RelationalExpression subject, ReferenceTypeName test) {
    checkNotNull(subject, "subject == null");
    checkNotNull(test, "test == null");
    return RelationalExpressionImpl.instanceOf0(subject, test);
  }

  // ShiftExpression
  
  public static ShiftExpression leftShift(ShiftExpression lhs, AdditiveExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return ShiftExpressionImpl.leftShift0(lhs, rhs);
  }
  
  public static ShiftExpression rightShift(ShiftExpression lhs, AdditiveExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return ShiftExpressionImpl.rightShift0(lhs, rhs);
  }
  
  public static ShiftExpression unsignedRightShift(ShiftExpression lhs, AdditiveExpression rhs) {
    checkNotNull(lhs, "lhs == null");
    checkNotNull(rhs, "rhs == null");
    return ShiftExpressionImpl.unsignedRightShift0(lhs, rhs);
  }
  
  // TypeWitness

  public static TypeWitness hint() {
    return witness();
  }

  public static TypeWitness hint(TypeName t1) {
    return witness(t1);
  }

  public static TypeWitness hint(TypeName t1, TypeName t2) {
    return witness(t1, t2);
  }

  public static TypeWitness hint(TypeName t1, TypeName t2, TypeName t3) {
    return witness(t1, t2, t3);
  }

  public static TypeWitness hint(Iterable<? extends TypeName> types) {
    return witness(types);
  }

  public static TypeWitness witness() {
    return TypeWitness.witness0();
  }

  public static TypeWitness witness(TypeName t1) {
    checkNotNull(t1, "t1 == null");
    return TypeWitness.witness0(t1);
  }

  public static TypeWitness witness(TypeName t1, TypeName t2) {
    checkNotNull(t1, "t1 == null");
    checkNotNull(t2, "t2 == null");
    return TypeWitness.witness0(t1, t2);
  }

  public static TypeWitness witness(TypeName t1, TypeName t2, TypeName t3) {
    checkNotNull(t1, "t1 == null");
    checkNotNull(t2, "t2 == null");
    checkNotNull(t3, "t3 == null");
    return TypeWitness.witness0(t1, t2, t3);
  }

  public static TypeWitness witness(Iterable<? extends TypeName> types) {
    checkNotNull(types, "types == null");
    return TypeWitness.witness0(types);
  }

  // UnaryExpression

  public static UnaryExpressionNotPlusMinus not(UnaryExpression expression) {
    checkNotNull(expression, "expression == null");
    return UnaryExpressionImpl.of0(UnaryOperator.NOT, expression);
  }

  public static UnaryExpression unaryMinus(UnaryExpression expression) {
    checkNotNull(expression, "expression == null");
    return UnaryExpressionImpl.of0(UnaryOperator.MINUS, expression);
  }

  public static UnaryExpression unaryPlus(UnaryExpression expression) {
    checkNotNull(expression, "expression == null");
    return UnaryExpressionImpl.of0(UnaryOperator.PLUS, expression);
  }

}
