/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.expression;

import br.com.objectos.code.java.element.AbstractDefaultCodeElement;
import br.com.objectos.code.java.element.ImmutableCodeElement;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.comuns.lang.Strings;

final class LiteralImpl extends AbstractDefaultCodeElement implements Literal {

  private LiteralImpl(ImmutableCodeElement delegate) {
    super(delegate);
  }

  static Literal l0(boolean value) {
    return ofPrimitive(Boolean.toString(value));
  }

  static Literal l0(char value) {
    return ofPrimitive("'" + Strings.escapeJava(Character.toString(value)) + "'");
  }

  static Literal l0(double value) {
    return ofPrimitive(Double.toString(value));
  }

  static Literal l0(int value) {
    return ofPrimitive(Integer.toString(value));
  }

  static Literal l0(float value) {
    return ofPrimitive(Float.toString(value) + "F");
  }

  static Literal l0(long value) {
    return ofPrimitive(Long.toString(value) + "L");
  }

  static Literal l0(Class<?> type) {
    return l0(ClassName.of(type));
  }

  static Literal l0(ClassName className) {
    return new LiteralImpl(
        ImmutableCodeElement.builder()
            .withTypeName(className)
            .withSpaceOff()
            .withWord(".class")
            .build()
    );
  }

  static Literal l0(String s) {
    return new LiteralImpl(
        ImmutableCodeElement.builder()
            .withStringLiteral(s)
            .build()
    );
  }

  private static Literal ofPrimitive(String value) {
    return new LiteralImpl(
        ImmutableCodeElement.builder()
            .withWord(value)
            .build()
    );
  }

  @Override
  public final FieldAccess id(Identifier id) {
    return Expressions.fieldAccess(this, id);
  }

  @Override
  public final FieldAccess id(String id) {
    return Expressions.fieldAccess(this, id);
  }

  @Override
  protected final ArrayReferenceExpression selfArrayReferenceExpression() {
    throw newUoe(Literal.class);
  }

  @Override
  protected final Callee selfCallee() {
    return this;
  }

  @Override
  protected final ConditionalAndExpression selfConditionalAndExpression() {
    return this;
  }

  @Override
  protected final LeftHandSide selfLeftHandSide() {
    throw newUoe(Literal.class);
  }

  @Override
  protected final MethodReferenceReferenceExpression selfMethodReferenceReferenceExpression() {
    return this;
  }

  @Override
  protected final MultiplicativeExpression selfMultiplicativeExpression() {
    return this;
  }

  @Override
  protected final PostfixExpression selfPostfixExpression() {
    return this;
  }

  @Override
  protected final RelationalExpression selfRelationalExpression() {
    return this;
  }

}
