/*
 * This file is part of the objectos :: code :: java project.
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package br.com.objectos.code.java.statement;

import br.com.objectos.code.java.expression.LambdaBody;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.io.Section;
import br.com.objectos.comuns.collections.GrowableList;
import br.com.objectos.comuns.collections.ImmutableList;

public class Block extends AbstractStatement implements LambdaBody {

  private final ImmutableList<? extends BlockElement> elements;

  private Block(ImmutableList<? extends BlockElement> elements) {
    this.elements = elements;
  }

  public static Block of(BlockElement... elements) {
    ImmutableList<BlockElement> list = ImmutableList.newListWithAll(elements);
    return new Block(list);
  }

  public static Block of(GrowableList<? extends BlockElement> elements) {
    return new Block(elements.toImmutableList());
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return elements.isEmpty()
        ? w.writeWord('{').write('}')
        : acceptCodeWriter0(w);
  }

  @Override
  public final void acceptSemicolon(Semicolon semicolon) {
    // noop
  }

  public final boolean isEmpty() {
    return elements.isEmpty();
  }

  private CodeWriter acceptCodeWriter0(CodeWriter w) {
    Semicolon semicolon = new Semicolon(w);

    w.writeWord('{');
    w.beginSection(Section.BLOCK);

    for (BlockElement statement : elements) {
      w.nextLine();
      w.beginSection(Section.STATEMENT);
      w.writeCodeElement(statement);
      w.endSection();
      statement.acceptSemicolon(semicolon);
    }

    w.nextLine();
    w.endSection();
    w.writeWord('}');
    return w;
  }

  public class Semicolon {

    private final CodeWriter w;

    private Semicolon(CodeWriter w) {
      this.w = w;
    }

    final void write() {
      w.spaceOff();
      w.write(';');
    }

  }

}
